<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\Form;

use Drupal\Component\Utility\Html;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Ajax\CloseModalDialogCommand;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\commerce_back_in_stock\Service\NotificationManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for Commerce Stock Notify entity forms (modal subscription).
 *
 * @ingroup commerce_back_in_stock
 */
class CommerceStockNotifyForm extends ContentEntityForm implements ContainerInjectionInterface {

  /**
   * Mail manager service.
   *
   * @var \Drupal\Core\Mail\MailManagerInterface
   */
  protected $mailManager;

  /**
   * Language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * Notification manager service.
   *
   * @var \Drupal\commerce_back_in_stock\Service\NotificationManager
   */
  protected $notificationManager;

  /**
   * Form constructor.
   *
   * @param \Drupal\Core\Mail\MailManagerInterface $mail_manager
   *   The mail manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Drupal\commerce_back_in_stock\Service\NotificationManager $notification_manager
   *   The notification manager service.
   */
  public function __construct(MailManagerInterface $mail_manager, LanguageManagerInterface $language_manager, ConfigFactoryInterface $config_factory, NotificationManager $notification_manager) {
    // Parent (FormBase) already has $this->configFactory property (untyped).
    $this->configFactory = $config_factory;
    $this->mailManager = $mail_manager;
    $this->languageManager = $language_manager;
    $this->notificationManager = $notification_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = new static(
      $container->get('plugin.manager.mail'),
      $container->get('language_manager'),
      $container->get('config.factory'),
      $container->get('commerce_back_in_stock.notification_manager'),
    );
    // Parent ContentEntityForm expects these properties from the container.
    $instance->entityRepository = $container->get('entity.repository');
    // Ensure time service is available for updateChangedTime().
    $instance->time = $container->get('datetime.time');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function init(FormStateInterface $form_state) {
    // Safety: ensure entity repository is available before parent init
    // calls initFormLangcodes().
    if (!isset($this->entityRepository)) {
      $this->entityRepository = \Drupal::service('entity.repository');
    }
    if (!isset($this->time)) {
      $this->time = \Drupal::service('datetime.time');
    }
    return parent::init($form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Ensure form has a valid entity instance before parent logic initializes
    // translation context and other internals.
    if (!isset($this->entity) || !$this->entity) {
      $entity = $this->getEntity();
      if (!$entity) {
        // Create a fresh entity if none was provided by the route.
        $entity = \Drupal::entityTypeManager()
          ->getStorage('commerce_back_in_stock')
          ->create();
        $this->setEntity($entity);
      }
    }

    /** @var \Drupal\commerce_back_in_stock\Entity\StockSubscription $entity */
    $form = parent::buildForm($form, $form_state);
    /**
     * @var CommerceStockNotifyForm $callback_object
*/
    $callback_object = $form_state->getBuildInfo()['callback_object'];
    $operation = $callback_object->getOperation();
    if ($operation === 'edit') {
      $form['product_id']['#disabled'] = TRUE;
    }
    elseif ($operation === 'add_page') {
      $form['product_id']['#disabled'] = TRUE;
      // Hide service/internal fields from the public modal form.
      if (isset($form['notified'])) {
        $form['notified']['#access'] = FALSE;
      }
    }

    // Alter field labels and placeholders from settings.
    $field_names = ['name', 'mail', 'phone', 'text'];
    foreach ($field_names as $field_name) {
      $label = $this->configFactory->get('commerce_back_in_stock.settings')
        ->get('form.labels.' . $field_name);
      if (!empty($label) && isset($form[$field_name]['widget'][0]['value'])) {
        // Move label text into placeholder to avoid overlay (themes with floating labels).
        $form[$field_name]['widget'][0]['value']['#placeholder'] = $label;
        // Keep accessible name even if visual label is hidden.
        $form[$field_name]['widget'][0]['value']['#attributes']['aria-label'] = $label;
        // Hide visual label entirely to prevent overlay.
        $form[$field_name]['widget'][0]['value']['#title'] = '';
        $form[$field_name]['widget'][0]['value']['#title_display'] = 'invisible';
        // Ensure no default value remains as a pseudo-placeholder ONLY for add_page.
        if ($operation === 'add_page' && isset($form[$field_name]['widget'][0]['value']['#default_value'])) {
          $form[$field_name]['widget'][0]['value']['#default_value'] = '';
        }
      }
    }

    // Improve UX for phone field: inputmode/aria and optional mask via JS.
    if (isset($form['phone']['widget'][0]['value'])) {
      $form['phone']['widget'][0]['value']['#attributes']['inputmode'] = 'tel';
      // Enable UA phone mask: +380 (XX) XXX-XX-XX
      $form['phone']['widget'][0]['value']['#attributes']['data-phone-mask'] = 'ua';
      $form['phone']['widget'][0]['value']['#attributes']['placeholder'] = '+380 (XX) XXX-XX-XX';
      $form['phone']['widget'][0]['value']['#attributes']['maxlength'] = 19;
      $form['phone']['widget'][0]['value']['#attributes']['pattern'] = '^\+380 \(\d{2}\) \d{3}-\d{2}-\d{2}$';
      $form['phone']['widget'][0]['value']['#attributes']['title'] = $this->t('Phone format: +380 (XX) XXX-XX-XX');
    }

    // Attach frontend behavior for phone mask and minor UX helpers.
    $form['#attached']['library'][] = 'commerce_back_in_stock/phone_form';

    // Add default value.
    if ($operation == 'add_page') {
      $current_user = $this->currentUser();
      if ($current_user->isAuthenticated()) {
        $form['name']['widget'][0]['value']['#default_value'] = $current_user->getDisplayName();
        $form['mail']['widget'][0]['value']['#default_value'] = $current_user->getEmail();
      }
    }

    $form['actions']['submit']['#value'] = $this->t('Submit');
    $label = $this->configFactory->get('commerce_back_in_stock.settings')->get('form.labels.submit');
    if (!empty($label)) {
      $form['actions']['submit']['#value'] = $label;
    }

    // ajaxSubmit.
    $form['#id'] = Html::getId($this->getFormId());
    $form['actions']['submit']['#ajax'] = [
      'callback' => '::ajaxSubmitCallback',
      'wrapper' => $form['#id'],
      'event' => 'click',
      'progress' => [
        'type' => 'throbber',
      ],
    ];

    return $form;
  }

  /**
   * AJAX callback for modal submission.
   *
   * Replaces the form with validation errors or success message and optionally
   * closes the modal dialog depending on configuration.
   *
   * @param array $form
   *   The render array of the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   The AJAX response with commands.
   */
  public function ajaxSubmitCallback(array &$form, FormStateInterface $form_state): AjaxResponse {
    $response = new AjaxResponse();
    $selector = '#' . $form['#id'];

    if ($form_state->hasAnyErrors()) {
      // Re-render the form inside the same wrapper to display
      // validation errors.
      unset($form['#prefix']);
      unset($form['#suffix']);
      $response->addCommand(new ReplaceCommand($selector, $form));
      return $response;
    }

    // Do NOT replace dialog content to avoid re-attaching legacy
    // jQuery UI dialog behaviors in some themes. Instead, show a status
    // message and (optionally) close the modal. This is the safest path
    // for Drupal 11 frontends.
    $submitted_text = (string) $this->configFactory->get('commerce_back_in_stock.settings')->get('form.submitted_text');
    if ($submitted_text !== '') {
      $response->addCommand(new \Drupal\Core\Ajax\MessageCommand($submitted_text, NULL, [
        'type' => 'status',
      ]));
    }

    // Optionally close the modal based on configuration.
    $autoClose = (bool) $this->configFactory
      ->get('commerce_back_in_stock.settings')
      ->get('popup.auto_close');
    if ($autoClose) {
      $response->addCommand(new CloseModalDialogCommand());
    }

    return $response;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Drupal\Core\Entity\EntityMalformedException
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $status = parent::save($form, $form_state);
    /**
     * @var StockSubscription $mynotify
*/
    $mynotify = $this->entity;
    $product = $mynotify->getProduct();
    if ($status == SAVED_UPDATED) {
      $this->messenger()->addStatus($this->t('The notify has been updated.'));
      $form_state->setRedirect('entity.commerce_back_in_stock.collection');
    }
    else {
      $this->messenger()->addStatus($this->t('Your message has been sent.'));
      // Delegate admin notification to the service (keeps token compatibility).
      $this->notificationManager->notifyAdminOnCreate($mynotify);
      $form_state->setRedirectUrl($product->toUrl());
    }
    return $status;
  }

}
