<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Settings form for Commerce Stock Notify.
 */
class CommerceStockNotifySettingsForm extends ConfigFormBase {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs the settings form.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
          $container->get('config.factory')
      );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'commerce_back_in_stock_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      'commerce_back_in_stock.settings',
      'commerce_back_in_stock.mail',
    ];
  }

  /**
   * Returns the field machine names used in the subscription form.
   *
   * @return string[]
   *   Array of field names.
   */
  protected function getFieldNames(): array {
    return [
      'name',
      'mail',
      'phone',
      'text',
      'submit',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $mail_config = $this->configFactory->get('commerce_back_in_stock.mail');
    $settings_config = $this->configFactory->get('commerce_back_in_stock.settings');
    $form['settings'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Settings'),
    ];
    // Popup form invocation.
    $form['load_popup'] = [
      '#type' => 'details',
      '#title' => $this->t('Popup invocation'),
      '#group' => 'settings',
      '#tree' => TRUE,
    ];
    $form['load_popup']['addtocart'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Replace button add to cart'),
      '#default_value' => $settings_config->get('load_popup.addtocart'),
      '#description' => $this->t('If the item is out of stock or the price is zero, replace the add to cart button with the pre-order button.'),
    ];

    // Popup options.
    $form['popup'] = [
      '#type' => 'details',
      '#title' => $this->t('Popup'),
      '#group' => 'settings',
      '#tree' => TRUE,
    ];
    $form['popup']['dialog_options'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Dialog options'),
      '#default_value' => $settings_config->get('popup.dialog_options'),
      '#size' => 120,
    ];
    $form['popup']['button_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Button name'),
      '#default_value' => $settings_config->get('popup.button_name'),
    ];
    $form['popup']['auto_close'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Close modal on success'),
      '#default_value' => (bool) $settings_config->get('popup.auto_close'),
      '#description' => $this->t('If enabled, the subscription modal will close automatically after a successful submission.'),
    ];
    $form['popup']['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#default_value' => $settings_config->get('popup.title'),
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['commerce_product'],
    ];
    // Add the token tree UI.
    $form['popup']['token_tree'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['commerce_product'],
      '#show_restricted' => TRUE,
      '#show_nested' => FALSE,
      '#weight' => 90,
    ];
    // Form labels and messages.
    $form['form'] = [
      '#type' => 'details',
      '#title' => $this->t('Form'),
      '#group' => 'settings',
      '#tree' => TRUE,
    ];
    $field_names = $this->getFieldNames();
    foreach ($field_names as $field_name) {
      $form['form']['labels'][$field_name] = [
        '#type' => 'textfield',
        '#title' => $this->t('Label for field:') . ' "' . $field_name . '"',
        '#default_value' => $settings_config->get('form.labels.' . $field_name),
        '#size' => 60,
      ];
    }
    $form['form']['submitted_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Success submission text'),
      '#default_value' => $settings_config->get('form.submitted_text'),
      '#rows' => 5,
    ];

    $form['email'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Email settings'),
    ];

    // Admin email.
    $form['admin_email'] = [
      '#type' => 'details',
      '#title' => $this->t('Admin email'),
      '#description' => $this->t('If left empty, admin emails will not be sent.'),
      '#group' => 'email',
    ];
    $form['admin_email']['admin_email'] = [
      '#type' => 'email',
      '#title' => $this->t('Email'),
      '#default_value' => $mail_config->get('admin.email'),
    ];
    // Admin email on new subscription request.
    $form['add_to_admin'] = [
      '#type' => 'details',
      '#title' => $this->t('Admin notification for new request'),
      '#description' => $this->t('A user requested to be notified when the product is back in stock.'),
      '#group' => 'email',
    ];
    $form['add_to_admin']['add_to_admin_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $mail_config->get('add_to_admin.subject'),
      '#maxlength' => 220,
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['mynotify'],
    ];
    $form['add_to_admin']['add_to_admin_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#default_value' => $mail_config->get('add_to_admin.body'),
      '#rows' => 15,
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['mynotify'],
    ];

    // Admin email when product is back in stock.
    $form['notified_add_to_admin'] = [
      '#type' => 'details',
      '#title' => $this->t('Admin notification: product in stock'),
      '#description' => $this->t('When a product becomes available, send an email to the admin with subscription details.'),
      '#group' => 'email',
    ];
    $form['notified_add_to_admin']['notified_add_to_admin_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $mail_config->get('notified_add_to_admin.subject'),
      '#maxlength' => 220,
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['mynotify'],
    ];
    $form['notified_add_to_admin']['notified_add_to_admin_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#default_value' => $mail_config->get('notified_add_to_admin.body'),
      '#rows' => 15,
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['mynotify'],
    ];
    // Customer email when product is back in stock.
    $form['notified_add_to_client'] = [
      '#type' => 'details',
      '#title' => $this->t('Customer notification: product in stock'),
      '#description' => $this->t('When a product becomes available, send emails to subscribers.'),
      '#group' => 'email',
    ];
    $form['notified_add_to_client']['notified_add_to_client_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $mail_config->get('notified_add_to_client.subject'),
      '#maxlength' => 220,
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['mynotify'],
    ];
    $form['notified_add_to_client']['notified_add_to_client_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Body'),
      '#default_value' => $mail_config->get('notified_add_to_client.body'),
      '#rows' => 15,
      '#element_validate' => ['token_element_validate'],
      '#token_types' => ['mynotify'],
    ];
    // Add the token tree UI.
    $form['email']['token_tree'] = [
      '#theme' => 'token_tree_link',
      '#token_types' => ['mynotify'],
      '#show_restricted' => TRUE,
      '#show_nested' => FALSE,
      '#weight' => 90,
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $popup = $form_state->getValue('popup');
    $dialog_options = $popup['dialog_options'];
    $dialog_options = Json::decode($dialog_options);
    if (!is_array($dialog_options)) {
      $form_state->setErrorByName('popup][dialog_options', $this->t('Dialog options: invalid JSON format'));
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Use editable configs for writes to avoid ImmutableConfigException.
    $mail_config = $this->configFactory->getEditable('commerce_back_in_stock.mail');
    $mail_config
      ->set('admin.email', $form_state->getValue('admin_email'))
      ->set('add_to_admin.subject', $form_state->getValue('add_to_admin_subject'))
      ->set('add_to_admin.body', $form_state->getValue('add_to_admin_body'))
      ->set('notified_add_to_client.subject', $form_state->getValue('notified_add_to_client_subject'))
      ->set('notified_add_to_client.body', $form_state->getValue('notified_add_to_client_body'))
      ->set('notified_add_to_admin.subject', $form_state->getValue('notified_add_to_admin_subject'))
      ->set('notified_add_to_admin.body', $form_state->getValue('notified_add_to_admin_body'));
    $mail_config->save();

    $settings_config = $this->configFactory->getEditable('commerce_back_in_stock.settings');
    $settings_config
      ->set('load_popup.addtocart', $form_state->getValue('load_popup')['addtocart'])
      ->set('popup.dialog_options', $form_state->getValue('popup')['dialog_options'])
      ->set('popup.button_name', $form_state->getValue('popup')['button_name'])
      ->set('popup.title', $form_state->getValue('popup')['title'])
      ->set('popup.auto_close', (bool) $form_state->getValue('popup')['auto_close']);
    $field_names = $this->getFieldNames();
    $labels = $form_state->getValue('form')['labels'];
    foreach ($field_names as $field_name) {
      $settings_config->set('form.labels.' . $field_name, $labels[$field_name]);
    }
    $settings_config->set('form.submitted_text', $form_state->getValue('form')['submitted_text']);
    $settings_config->save();
    parent::submitForm($form, $form_state);
  }

}
