<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\Form;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Core\Pager\PagerManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Bulk operations form for Stock Subscriptions.
 */
class StockSubscriptionBulkListForm extends FormBase implements ContainerInjectionInterface {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected DateFormatterInterface $dateFormatter,
    protected PagerManagerInterface $pagerManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('date.formatter'),
      $container->get('pager.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'commerce_back_in_stock_bulk_list_form';
  }

  /**
   * Loads a limited set of subscriptions for display.
   *
   * @return array
   *   Array of entities keyed by ID.
   */
  protected function loadRows(string $filter_name = '', int $limit = 25): array {
    $storage = $this->entityTypeManager->getStorage('commerce_back_in_stock');

    // Build total query (for pager) with optional filters.
    $count_query = \Drupal::entityQuery('commerce_back_in_stock')->accessCheck(FALSE);
    if ($filter_name !== '') {
      $count_query->condition('name', '%' . $filter_name . '%', 'LIKE');
    }
    $total = (int) $count_query->count()->execute();

    // Initialize the pager and compute the offset for the page.
    $pager = $this->pagerManager->createPager($total, $limit);
    $current_page = (int) $pager->getCurrentPage();
    $offset = $current_page * $limit;

    // Load the current page of IDs with the same filters and sort.
    $id_query = \Drupal::entityQuery('commerce_back_in_stock')->accessCheck(FALSE)
      ->sort('id', 'DESC')
      ->range($offset, $limit);
    if ($filter_name !== '') {
      $id_query->condition('name', '%' . $filter_name . '%', 'LIKE');
    }
    $ids = $id_query->execute();
    if (empty($ids)) {
      return [];
    }
    return $storage->loadMultiple($ids);
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    // Filters block (exposed filter by name, stored in query string).
    $current_request = \Drupal::service('request_stack')->getCurrentRequest();
    $current_name = (string) ($current_request?->query->get('name') ?? '');
    $form['filters'] = [
      '#type' => 'details',
      '#title' => $this->t('Filters'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];
    $form['filters']['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name contains'),
      '#default_value' => $current_name,
      '#size' => 32,
    ];
    $form['filters']['actions'] = [
      '#type' => 'actions',
      'apply' => [
        '#type' => 'submit',
        '#value' => $this->t('Apply filters'),
        '#submit' => ['::submitApplyFilters'],
      ],
      'reset' => [
        '#type' => 'submit',
        '#value' => $this->t('Reset'),
        '#submit' => ['::submitResetFilters'],
      ],
    ];

    $form['operations'] = [
      '#type' => 'actions',
      'mark_notified' => [
        '#type' => 'submit',
        '#value' => $this->t('Mark notified'),
        '#submit' => ['::submitMarkNotified'],
      ],
      'mark_pending' => [
        '#type' => 'submit',
        '#value' => $this->t('Mark pending'),
        '#submit' => ['::submitMarkPending'],
      ],
      'delete' => [
        '#type' => 'submit',
        '#value' => $this->t('Delete selected'),
        '#submit' => ['::submitDeleteSelected'],
        '#attributes' => ['class' => ['button--danger']],
      ],
    ];

    $form['table'] = [
      '#type' => 'table',
      '#header' => [
        'select' => ['data' => $this->t('Select')],
        'id' => ['data' => $this->t('ID')],
        'name' => ['data' => $this->t('Name')],
        'mail' => ['data' => $this->t('Mail')],
        'product' => ['data' => $this->t('Product')],
        'notified' => ['data' => $this->t('Notified')],
        'created' => ['data' => $this->t('Created')],
        'operations' => ['data' => $this->t('Operations')],
      ],
      '#empty' => $this->t('No subscriptions found.'),
    ];

    $rows = $this->loadRows($current_name, 25);
    foreach ($rows as $entity) {
      /** @var \Drupal\commerce_back_in_stock\Entity\StockSubscription $entity */
      $id = (int) $entity->id();
      $product = $entity->getProduct();
      $productLink = $product ? $product->toLink() : Link::fromTextAndUrl($this->t('N/A'), Url::fromRoute('<none>'));
      $form['table'][$id]['select'] = [
        '#type' => 'checkbox',
      ];
      $form['table'][$id]['id'] = ['#markup' => (string) $id];
      $form['table'][$id]['name'] = ['#markup' => $entity->getName() ?: '-'];
      $form['table'][$id]['mail'] = ['#markup' => $entity->getMail() ?: '-'];
      $form['table'][$id]['product'] = $productLink->toRenderable();
      $form['table'][$id]['notified'] = ['#markup' => $entity->isNotified() ? $this->t('Yes') : $this->t('No')];
      $created = (int) $entity->get('created')->value;
      $form['table'][$id]['created'] = ['#markup' => $this->dateFormatter->format($created, 'medium')];
      $form['table'][$id]['operations'] = [
        '#type' => 'operations',
        '#links' => [
          'edit' => [
            'title' => $this->t('Edit'),
            'url' => Url::fromRoute('entity.commerce_back_in_stock.edit_form', ['commerce_back_in_stock' => $id]),
          ],
          'delete' => [
            'title' => $this->t('Delete'),
            'url' => Url::fromRoute('entity.commerce_back_in_stock.delete_form', ['commerce_back_in_stock' => $id]),
          ],
        ],
      ];
    }

    // Select all checkbox.
    $form['table']['#sticky'] = TRUE;
    $form['table']['#attributes']['class'][] = 'responsive-enabled';

    // Add pager element for navigation.
    $form['pager'] = [
      '#type' => 'pager',
    ];

    return $form;
  }

  /**
   * Gets selected entity IDs from the form state.
   */
  protected function getSelectedIds(array $form, FormStateInterface $form_state): array {
    $values = $form_state->getValue('table') ?? [];
    $selected = [];
    foreach ($values as $id => $row) {
      if (!empty($row['select'])) {
        $selected[] = (int) $id;
      }
    }
    return $selected;
  }

  /**
   * Submit handler: mark selected as notified.
   */
  public function submitMarkNotified(array &$form, FormStateInterface $form_state): void {
    $ids = $this->getSelectedIds($form, $form_state);
    if (empty($ids)) {
      $this->messenger()->addWarning($this->t('No items selected.'));
      return;
    }
    $storage = $this->entityTypeManager->getStorage('commerce_back_in_stock');
    $entities = $storage->loadMultiple($ids);
    $count = 0;
    foreach ($entities as $entity) {
      $entity->setNotified(TRUE);
      $entity->save();
      $count++;
    }
    $this->messenger()->addStatus($this->t('Marked @count subscription(s) as notified.', ['@count' => $count]));
    $form_state->setRebuild();
  }

  /**
   * Submit handler: mark selected as pending (not notified).
   */
  public function submitMarkPending(array &$form, FormStateInterface $form_state): void {
    $ids = $this->getSelectedIds($form, $form_state);
    if (empty($ids)) {
      $this->messenger()->addWarning($this->t('No items selected.'));
      return;
    }
    $storage = $this->entityTypeManager->getStorage('commerce_back_in_stock');
    $entities = $storage->loadMultiple($ids);
    $count = 0;
    foreach ($entities as $entity) {
      $entity->setNotified(FALSE);
      $entity->save();
      $count++;
    }
    $this->messenger()->addStatus($this->t('Marked @count subscription(s) as pending.', ['@count' => $count]));
    $form_state->setRebuild();
  }

  /**
   * Submit handler: delete selected items.
   */
  public function submitDeleteSelected(array &$form, FormStateInterface $form_state): void {
    $ids = $this->getSelectedIds($form, $form_state);
    if (empty($ids)) {
      $this->messenger()->addWarning($this->t('No items selected.'));
      return;
    }
    $storage = $this->entityTypeManager->getStorage('commerce_back_in_stock');
    $entities = $storage->loadMultiple($ids);
    $count = 0;
    foreach ($entities as $entity) {
      $entity->delete();
      $count++;
    }
    $this->messenger()->addStatus($this->t('Deleted @count subscription(s).', ['@count' => $count]));
    $form_state->setRebuild();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    // Not used. Individual submit handlers are attached to buttons above.
  }

  /**
   * Submit handler: apply filters via query string redirect.
   */
  public function submitApplyFilters(array &$form, FormStateInterface $form_state): void {
    $name = (string) $form_state->getValue(['filters', 'name']);
    $options = [];
    if ($name !== '') {
      $options['query']['name'] = $name;
    }
    $form_state->setRedirect('entity.commerce_back_in_stock.collection', [], $options);
  }

  /**
   * Submit handler: reset filters.
   */
  public function submitResetFilters(array &$form, FormStateInterface $form_state): void {
    $form_state->setRedirect('entity.commerce_back_in_stock.collection');
  }

}
