<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\Plugin\QueueWorker;

use Drupal\commerce_back_in_stock\Entity\StockSubscription;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Sends admin notifications when a product becomes available.
 *
 * @QueueWorker(
 *   id = "commerce_back_in_stock_admin",
 *   title = @Translation("Commerce Back in Stock: Admin notifications"),
 *   cron = {"time" = 30}
 * )
 */
class AdminStockNoticeWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs the queue worker plugin instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Mail\MailManagerInterface $mailManager
   *   The mail manager.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger channel factory.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected MailManagerInterface $mailManager,
    protected LanguageManagerInterface $languageManager,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelFactoryInterface $loggerFactory,
  ) {
  }

  /**
   * {@inheritdoc}
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   * @param array $configuration
   *   Plugin configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
          $container->get('entity_type.manager'),
          $container->get('plugin.manager.mail'),
          $container->get('language_manager'),
          $container->get('config.factory'),
          $container->get('logger.factory'),
      );
  }

  /**
   * {@inheritdoc}
   *
   * @param array $data
   *   Queue data with keys: product_id, sample_subscription_id (optional),
   *   count (optional).
   */
  public function processItem($data): void {
    $logger = $this->loggerFactory->get('commerce_back_in_stock');
    if (!is_array($data) || empty($data['product_id'])) {
      $logger->warning('Admin queue item is invalid: @data', ['@data' => print_r($data, TRUE)]);
      return;
    }

    $mail_config = $this->configFactory->get('commerce_back_in_stock.mail');
    $to = (string) $mail_config->get('admin.email');
    if ($to === '') {
      // Admin email not configured.
      return;
    }

    $subscription = NULL;
    if (!empty($data['sample_subscription_id'])) {
      $sid = (int) $data['sample_subscription_id'];
      $loaded = $this->entityTypeManager->getStorage('commerce_back_in_stock')->load($sid);
      if ($loaded instanceof StockSubscription) {
        $subscription = $loaded;
      }
    }

    $params = [
      'body' => $mail_config->get('notified_add_to_admin.body'),
      'subject' => $mail_config->get('notified_add_to_admin.subject'),
      // Token context variable kept for backward compatibility.
      'mynotify' => $subscription,
      // Extra info for potential use in templates later.
      'count' => isset($data['count']) ? (int) $data['count'] : NULL,
      'product_id' => (int) $data['product_id'],
    ];
    $langcode = $this->languageManager->getCurrentLanguage()->getId();
    $result = $this->mailManager->mail('commerce_back_in_stock', 'notified_add_to_admin', $to, $langcode, $params);
    if (empty($result['result'])) {
      $logger->error(
            'Failed to send admin stock notice for product @pid to @mail.',
            [
              '@pid' => $data['product_id'],
              '@mail' => $to,
            ]
        );
      throw new \RuntimeException('Failed to send admin mail');
    }
  }

}
