<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\Plugin\QueueWorker;

use Drupal\commerce_back_in_stock\Entity\StockSubscription;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Processes queued customer stock notifications.
 *
 * @QueueWorker(
 *   id = "commerce_back_in_stock_customer",
 *   title = @Translation("Commerce Back in Stock: Customer notifications"),
 *   cron = {"time" = 30}
 * )
 */
class StockNotificationWorker extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected MailManagerInterface $mailManager,
    protected LanguageManagerInterface $languageManager,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerChannelFactoryInterface $loggerFactory,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
          $container->get('entity_type.manager'),
          $container->get('plugin.manager.mail'),
          $container->get('language_manager'),
          $container->get('config.factory'),
          $container->get('logger.factory'),
      );
  }

  /**
   * {@inheritdoc}
   */
  public function processItem($data): void {
    $logger = $this->loggerFactory->get('commerce_back_in_stock');
    if (!is_array($data) || empty($data['subscription_id'])) {
      $logger->warning('Queue item is invalid: @data', ['@data' => print_r($data, TRUE)]);
      return;
    }
    $sid = (int) $data['subscription_id'];
    /**
     * @var \Drupal\commerce_back_in_stock\Entity\StockSubscription|null $subscription
     */
    $subscription = $this->entityTypeManager->getStorage('commerce_back_in_stock')->load($sid);
    if (!$subscription instanceof StockSubscription) {
      $logger->warning('Subscription @id not found; skipping.', ['@id' => $sid]);
      return;
    }

    // If already notified, skip.
    if ($subscription->isNotified()) {
      return;
    }

    $mail = $subscription->getMail();
    if (!$mail) {
      $logger->warning('Subscription @id has no email; skipping.', ['@id' => $sid]);
      return;
    }

    $mail_config = $this->configFactory->get('commerce_back_in_stock.mail');
    $params = [
      'body' => $mail_config->get('notified_add_to_client.body'),
      'subject' => $mail_config->get('notified_add_to_client.subject'),
      // Token context compatibility variable name.
      'mynotify' => $subscription,
    ];
    $langcode = $this->languageManager->getCurrentLanguage()->getId();
    $result = $this->mailManager->mail('commerce_back_in_stock', 'notified_add_to_client', $mail, $langcode, $params);
    if (!empty($result['result'])) {
      $subscription->setNotified(TRUE);
      $subscription->save();
    }
    else {
      // Throwing signals a failure; the item may be retried depending on
      // the queue backend configuration.
      $logger->error(
            'Failed to send customer notification for subscription @id to @mail.',
            [
              '@id' => $sid,
              '@mail' => $mail,
            ]
        );
      throw new \RuntimeException('Failed to send mail');
    }
  }

}
