<?php

declare(strict_types=1);

namespace Drupal\commerce_back_in_stock\Service;

use Drupal\commerce_back_in_stock\Entity\StockSubscription;
use Drupal\commerce_product\Entity\Product;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Queue\QueueFactory;

/**
 * Centralized notification logic for Commerce Back in Stock.
 */
class NotificationManager {

  /**
   * Queue IDs used by this module.
   */
  public const QUEUE_CUSTOMER = 'commerce_back_in_stock_customer';
  public const QUEUE_ADMIN = 'commerce_back_in_stock_admin';

  /**
   * Module logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  public function __construct(
    protected MailManagerInterface $mailManager,
    protected LanguageManagerInterface $languageManager,
    protected ConfigFactoryInterface $configFactory,
    LoggerChannelFactoryInterface $loggerFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModuleHandlerInterface $moduleHandler,
    protected QueueFactory $queueFactory,
  ) {
    // Obtain the module-specific logger channel from the factory.
    $this->logger = $loggerFactory->get('commerce_back_in_stock');
  }

  /**
   * Sends admin notification when a new subscription is created.
   */
  public function notifyAdminOnCreate(StockSubscription $subscription): void {
    $mail_config = $this->configFactory->get('commerce_back_in_stock.mail');
    $to = (string) $mail_config->get('admin.email');
    if ($to === '') {
      return;
    }
    $module = 'commerce_back_in_stock';
    $key = 'add_to_admin';
    $langcode = $this->languageManager->getCurrentLanguage()->getId();
    $params = [
      'subject' => $mail_config->get('add_to_admin.subject'),
      'body' => $mail_config->get('add_to_admin.body'),
      // Backward-compatible token variable name retained as 'mynotify'.
      'mynotify' => $subscription,
    ];
    $result = $this->mailManager->mail($module, $key, $to, $langcode, $params);
    if (empty($result['result'])) {
      $this->logger->warning('Failed to send admin notification email for subscription @id.', ['@id' => $subscription->id()]);
    }
  }

  /**
   * Checks product availability with legacy alter for compatibility.
   */
  public function isProductAvailable(Product $product): bool {
    $is_availability = FALSE;
    if (method_exists($product, 'isAvailability')) {
      $is_availability = (bool) $product->isAvailability();
    }
    // Backward-compatible alter name.
    $this->moduleHandler->alter('mynotify_is_availability', $product, $is_availability);
    return $is_availability;
  }

  /**
   * Handle product update and enqueue notifications if product is available.
   */
  public function onProductUpdate(Product $product): void {
    if (!$this->isProductAvailable($product)) {
      return;
    }
    $storage = $this->entityTypeManager->getStorage('commerce_back_in_stock');
    if (!method_exists($storage, 'loadPendingNotificationIds')) {
      return;
    }
    /**
     * @var int[] $ids
*/
    $ids = $storage->loadPendingNotificationIds($product->id());
    if (empty($ids)) {
      return;
    }

    // Enqueue customer notifications by IDs in chunks.
    // No need to load entity objects here.
    $customerQueue = $this->queueFactory->get(self::QUEUE_CUSTOMER);
    $count = 0;
    $chunkSize = 100;
    foreach (array_chunk($ids, $chunkSize) as $chunk) {
      foreach ($chunk as $sid) {
        $customerQueue->createItem(['subscription_id' => (int) $sid]);
        $count++;
      }
    }
    $this->logger->info(
          'Queued @count stock notifications for product @product.',
          [
            '@count' => $count,
            '@product' => $product->id(),
          ]
      );

    // Enqueue a single admin notice with a sample subscription id for
    // token context.
    $sampleId = (int) reset($ids);
    $adminQueue = $this->queueFactory->get(self::QUEUE_ADMIN);
    $adminQueue->createItem(
          [
            'product_id' => (int) $product->id(),
            'sample_subscription_id' => $sampleId,
            'count' => $count,
          ]
      );
  }

}
