<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_back_in_stock\Kernel;

use Drupal\KernelTests\KernelTestBase;

/**
 * Kernel tests for StockSubscription storage queries.
 *
 * @group commerce_back_in_stock
 */
class StockSubscriptionStorageKernelTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'text',
    'options',
    'telephone',
    'commerce',
    'commerce_product',
    'commerce_back_in_stock',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    // Install entity schemas for our content entity and user (owner field).
    $this->installEntitySchema('user');
    $this->installEntitySchema('commerce_back_in_stock');
    // Some content entities rely on sequences; ensure system schemas.
    $this->installSchema('system', ['sequences']);
  }

  /**
   * Ensures pending IDs are filtered by product and notified flag.
   */
  public function testLoadPendingNotificationIdsFiltersByProductAndNotified(): void {
    $storage = $this->container->get('entity_type.manager')->getStorage('commerce_back_in_stock');

    // Create three subscriptions: two pending for product 1, one already notified.
    $e1 = $storage->create(
          [
            'product_id' => 1,
            'name' => 'User 1',
            'mail' => 'u1@example.com',
            'notified' => 0,
          ]
      );
    $e1->save();
    $e2 = $storage->create(
          [
            'product_id' => 1,
            'name' => 'User 2',
            'mail' => 'u2@example.com',
            'notified' => 0,
          ]
      );
    $e2->save();
    $e3 = $storage->create(
          [
            'product_id' => 1,
            'name' => 'User 3',
            'mail' => 'u3@example.com',
            'notified' => 1,
          ]
      );
    $e3->save();
    // Different product, pending.
    $e4 = $storage->create(
          [
            'product_id' => 2,
            'name' => 'User 4',
            'mail' => 'u4@example.com',
            'notified' => 0,
          ]
      );
    $e4->save();

    // By product 1: only two IDs, and they must be integers.
    $ids = $storage->loadPendingNotificationIds(1);
    sort($ids);
    $this->assertSame([(int) $e1->id(), (int) $e2->id()], $ids);

    // By product 2: one ID (integer).
    $ids2 = $storage->loadPendingNotificationIds(2);
    $this->assertSame([(int) $e4->id()], $ids2);

    // Without product filter: three pending across all products (integers).
    $idsAll = $storage->loadPendingNotificationIds();
    sort($idsAll);
    $this->assertSame([(int) $e1->id(), (int) $e2->id(), (int) $e4->id()], $idsAll);
  }

}
