<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_back_in_stock\Unit;

use Drupal\commerce_back_in_stock\Plugin\QueueWorker\AdminStockNoticeWorker;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Mail\MailManagerInterface;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for AdminStockNoticeWorker queue processor.
 */
class AdminStockNoticeWorkerTest extends TestCase {

  /**
   * Skips processing when admin email is not configured.
   */
  public function testProcessItemSkipsWhenAdminEmailEmpty(): void {
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $mailManager = $this->createMock(MailManagerInterface::class);
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);

    // Config returns empty admin email.
    $config = $this->createMock(Config::class);
    $config->method('get')->with('admin.email')->willReturn('');
    $configFactory->method('get')->with('commerce_back_in_stock.mail')->willReturn($config);

    // Language manager current language.
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getId')->willReturn('en');
    $languageManager->method('getCurrentLanguage')->willReturn($language);

    // Logger factory not used in this path, but provide default.
    $logger = $this->createMock(LoggerChannelInterface::class);
    $loggerFactory->method('get')->with('commerce_back_in_stock')->willReturn($logger);

    $worker = new AdminStockNoticeWorker(
          $entityTypeManager,
          $mailManager,
          $languageManager,
          $configFactory,
          $loggerFactory,
      );

    // Mail manager should NOT be called when admin email is empty.
    $mailManager->expects($this->never())->method('mail');

    $worker->processItem(['product_id' => 1]);
    // If no exception and no mail call, test passes.
    $this->assertTrue(TRUE);
  }

  /**
   * Logs warning on invalid queue item payload.
   */
  public function testProcessItemWarnsOnInvalidData(): void {
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $mailManager = $this->createMock(MailManagerInterface::class);
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);

    $logger = $this->createMock(LoggerChannelInterface::class);
    $logger->expects($this->once())->method('warning');
    $loggerFactory->method('get')->with('commerce_back_in_stock')->willReturn($logger);

    $worker = new AdminStockNoticeWorker(
          $entityTypeManager,
          $mailManager,
          $languageManager,
          $configFactory,
          $loggerFactory,
      );

    $worker->processItem(['foo' => 'bar']);
    $this->assertTrue(TRUE);
  }

}
