<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_back_in_stock\Unit;

use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_back_in_stock\Service\NotificationManager;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Queue\QueueFactory;
use Drupal\Core\Queue\QueueInterface;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for NotificationManager service.
 */
class NotificationManagerTest extends TestCase {

  /**
   * Ensures product update enqueues customer and admin notifications.
   */
  public function testOnProductUpdateQueuesItemsAndAdminNotice(): void {
    // Product mock: isAvailability() => true.
    /**
     * @var \PHPUnit\Framework\MockObject\MockObject|Product $product
*/
    $product = $this->getMockBuilder(Product::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['id'])
      ->addMethods(['isAvailability'])
      ->getMock();
    $product->method('isAvailability')->willReturn(TRUE);
    $product->method('id')->willReturn(42);

    // Provide a lightweight storage stub that has the expected method.
    $storage = new class {

      /**
       * Returns a fixed set of pending subscription IDs for the given product.
       *
       * @param int $pid
       *   The product ID.
       *
       * @return int[]
       *   An array of subscription IDs.
       */
      public function loadPendingNotificationIds($pid): array {
        return [10, 11, 12];
      }

    };

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')->with('commerce_back_in_stock')->willReturn($storage);

    // Queues: customer and admin.
    $customerQueue = $this->createMock(QueueInterface::class);
    $adminQueue = $this->createMock(QueueInterface::class);
    // Expect 3 createItem calls for customer queue.
    $customerQueue->expects($this->exactly(3))
      ->method('createItem')
      ->with(
              $this->callback(
                  function ($payload) {
                      return is_array($payload) && isset($payload['subscription_id']);
                  }
              )
          );
    // Expect one admin createItem call.
    $adminQueue->expects($this->once())
      ->method('createItem')
      ->with(
              $this->callback(
                  function ($payload) {
                      return isset($payload['product_id'], $payload['sample_subscription_id'], $payload['count'])
                      && $payload['product_id'] === 42 && $payload['count'] === 3;
                  }
              )
          );

    $queueFactory = $this->createMock(QueueFactory::class);
    $queueFactory->method('get')->willReturnMap(
          [
          ['commerce_back_in_stock_customer', $customerQueue],
          ['commerce_back_in_stock_admin', $adminQueue],
          ]
      );

    $mailManager = $this->createMock(MailManagerInterface::class);

    $language = $this->createMock(LanguageInterface::class);
    $language->method('getId')->willReturn('en');
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getCurrentLanguage')->willReturn($language);

    $config = $this->createMock(Config::class);
    $config->method('get')->willReturn(NULL);
    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory->method('get')->willReturn($config);

    // Logger factory returns our channel for the module name.
    $loggerChannel = $this->createMock(LoggerChannelInterface::class);
    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $loggerFactory->method('get')->with('commerce_back_in_stock')->willReturn($loggerChannel);
    // We don't assert logs strictly here.
    $moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    // No hook alters expected; just ensure alter() callable doesn't break.
    $moduleHandler->method('alter');

    $manager = new NotificationManager(
          $mailManager,
          $languageManager,
          $configFactory,
          $loggerFactory,
          $entityTypeManager,
          $moduleHandler,
          $queueFactory,
      );

    $manager->onProductUpdate($product);
    $this->assertTrue(TRUE);
  }

  /**
   * Ensures early returns when unavailable or when there are no IDs.
   */
  public function testOnProductUpdateSkipsWhenUnavailableOrEmptyIds(): void {
    /**
     * @var \PHPUnit\Framework\MockObject\MockObject|Product $product
*/
    $product = $this->getMockBuilder(Product::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['id'])
      ->addMethods(['isAvailability'])
      ->getMock();
    $product->method('isAvailability')->willReturn(FALSE);
    $product->method('id')->willReturn(7);

    // Provide a lightweight storage stub for negative branches.
    $storage = new class {

      /**
       * Returns an empty set of subscription IDs (nothing pending).
       *
       * @param int $pid
       *   The product ID.
       *
       * @return int[]
       *   An empty array.
       */
      public function loadPendingNotificationIds($pid): array {
        return [];
      }

    };
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')->willReturn($storage);

    $queueFactory = $this->createMock(QueueFactory::class);
    // Queues should not be obtained when unavailable.
    $queueFactory->expects($this->never())->method('get');

    $mailManager = $this->createMock(MailManagerInterface::class);
    $language = $this->createMock(LanguageInterface::class);
    $language->method('getId')->willReturn('en');
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getCurrentLanguage')->willReturn($language);
    $config = $this->createMock(Config::class);
    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory->method('get')->willReturn($config);
    $loggerChannel = $this->createMock(LoggerChannelInterface::class);
    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $loggerFactory->method('get')->with('commerce_back_in_stock')->willReturn($loggerChannel);
    $moduleHandler = $this->createMock(ModuleHandlerInterface::class);

    $manager = new NotificationManager(
          $mailManager,
          $languageManager,
          $configFactory,
          $loggerFactory,
          $entityTypeManager,
          $moduleHandler,
          $queueFactory,
      );
    // Should early return due to unavailability.
    $manager->onProductUpdate($product);

    // Now simulate available but no IDs.
    $product2 = $this->getMockBuilder(Product::class)
      ->disableOriginalConstructor()
      ->onlyMethods(['id'])
      ->addMethods(['isAvailability'])
      ->getMock();
    $product2->method('isAvailability')->willReturn(TRUE);
    $product2->method('id')->willReturn(8);
    // Swap storage stub to one that returns an empty list for the available case.
    $storage2 = new class {

      /**
       * Returns an empty set of subscription IDs (nothing pending).
       *
       * @param int $pid
       *   The product ID.
       *
       * @return int[]
       *   An empty array.
       */
      public function loadPendingNotificationIds($pid): array {
        return [];
      }

    };
    $entityTypeManager->method('getStorage')->with('commerce_back_in_stock')->willReturn($storage2);
    // Still no queues should be called when empty list.
    $queueFactory->expects($this->never())->method('get');

    $manager->onProductUpdate($product2);
    $this->assertTrue(TRUE);
  }

}
