<?php

declare(strict_types=1);

namespace Drupal\Tests\commerce_back_in_stock\Unit;

use Drupal\commerce_back_in_stock\Entity\StockSubscription;
use Drupal\commerce_back_in_stock\Plugin\QueueWorker\StockNotificationWorker;
use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Mail\MailManagerInterface;
use PHPUnit\Framework\TestCase;

/**
 * Unit tests for StockNotificationWorker queue processor.
 */
class StockNotificationWorkerTest extends TestCase {

  /**
   * Marks subscription as notified when mail sending succeeds.
   */
  public function testProcessItemMarksSubscriptionAsNotifiedOnSuccess(): void {
    $subscription = $this->createMock(StockSubscription::class);
    $subscription->method('isNotified')->willReturn(FALSE);
    $subscription->method('getMail')->willReturn('user@example.com');
    $subscription->expects($this->once())->method('setNotified')->with(TRUE)->willReturnSelf();
    $subscription->expects($this->once())->method('save');

    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('load')->with(123)->willReturn($subscription);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')->with('commerce_back_in_stock')->willReturn($storage);

    $mailManager = $this->createMock(MailManagerInterface::class);
    $mailManager->method('mail')->willReturn(['result' => TRUE]);

    $language = $this->createMock(LanguageInterface::class);
    $language->method('getId')->willReturn('en');

    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $languageManager->method('getCurrentLanguage')->willReturn($language);

    $config = $this->createMock(Config::class);
    $config->method('get')->willReturnMap(
          [
          ['notified_add_to_client.body', 'Body'],
          ['notified_add_to_client.subject', 'Subject'],
          ]
      );

    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $configFactory->method('get')->with('commerce_back_in_stock.mail')->willReturn($config);

    $logger = $this->createMock(LoggerChannelInterface::class);
    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);
    $loggerFactory->method('get')->with('commerce_back_in_stock')->willReturn($logger);

    $worker = new StockNotificationWorker(
          $entityTypeManager,
          $mailManager,
          $languageManager,
          $configFactory,
          $loggerFactory,
      );

    $worker->processItem(['subscription_id' => 123]);
    $this->assertTrue(TRUE);
  }

  /**
   * Skips sending when subscription is already notified.
   */
  public function testProcessItemSkipsIfAlreadyNotified(): void {
    $subscription = $this->createMock(StockSubscription::class);
    $subscription->method('isNotified')->willReturn(TRUE);

    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('load')->with(1)->willReturn($subscription);

    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $entityTypeManager->method('getStorage')->with('commerce_back_in_stock')->willReturn($storage);

    $mailManager = $this->createMock(MailManagerInterface::class);
    $languageManager = $this->createMock(LanguageManagerInterface::class);
    $configFactory = $this->createMock(ConfigFactoryInterface::class);
    $loggerFactory = $this->createMock(LoggerChannelFactoryInterface::class);

    $worker = new StockNotificationWorker(
          $entityTypeManager,
          $mailManager,
          $languageManager,
          $configFactory,
          $loggerFactory,
      );

    // Mail should not be sent when already notified.
    $mailManager->expects($this->never())->method('mail');

    $worker->processItem(['subscription_id' => 1]);
    $this->assertTrue(TRUE);
  }

}
