<?php

namespace Drupal\commerce_barion_payment\Plugin\Commerce\PaymentGateway;

use Barion\Enumerations\BarionEnvironment;
use Barion\BarionClient;
use Barion\Enumerations\Currency;
use Barion\Enumerations\FundingSourceType;
use Barion\Enumerations\PaymentType;
use Barion\Enumerations\UILocale;
use Barion\Models\Common\ItemModel;
use Barion\Models\Payment\FinishReservationRequestModel;
use Barion\Models\Payment\PaymentTransactionModel;
use Barion\Models\Payment\PreparePaymentRequestModel;
use Barion\Models\Payment\TransactionToFinishModel;
use Barion\Models\ThreeDSecure\BillingAddressModel;
use Drupal\commerce_barion_payment\PluginForm\BarionRedirectForm;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Attribute\CommercePaymentGateway;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsAuthorizationsInterface;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Barion off-site Redirect payment gateway.
 */
#[CommercePaymentGateway(
  id: "barion_payment",
  label: new TranslatableMarkup("Barion"),
  display_label: new TranslatableMarkup("Barion"),
  forms: [
    "offsite-payment" => BarionRedirectForm::class,
  ],
)]
class BarionPaymentGateway extends OffsitePaymentGatewayBase implements SupportsAuthorizationsInterface {

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->logger = $container->get('logger.factory')->get('commerce_barion_payment');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'email' => '',
      'private_key' => '',
      'api_version' => '2',
      'payment_window' => '00:05:00',
      'locale' => UILocale::EN->value,
      'reservation_period' => '0.00:30:00',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {

    $form = parent::buildConfigurationForm($form, $form_state);

    $form['email'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Barion email address'),
      '#maxlength' => 64,
      '#size' => 64,
      '#weight' => '0',
      '#required' => TRUE,
      '#default_value' => $this->configuration['email'],
    ];

    $form['private_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Secret key (POSKey)'),
      '#description' => $this->t('Secret key provided by the Barion service.'),
      '#maxlength' => 64,
      '#size' => 64,
      '#weight' => '0',
      '#required' => TRUE,
      '#default_value' => $this->configuration['private_key'],
    ];

    $form['api_version'] = [
      '#type' => 'number',
      '#title' => $this->t('API version number'),
      '#maxlength' => 64,
      '#size' => 64,
      '#weight' => '0',
      '#required' => TRUE,
      '#default_value' => $this->configuration['api_version'],
    ];

    $payment_window_parts = explode(':', $this->configuration['payment_window']);
    $form['payment_window_label_item'] = [
      '#type' => 'item',
      '#title' => $this->t('Payment Window(HMS)'),
      'payment_window' => [
        '#type' => 'container',
        '#attributes' => [
          'class' => [
            'container-inline',
          ],
        ],
        0 => [
          '#type' => 'textfield',
          '#default_value' => $payment_window_parts[0],
          '#size' => 2,
        ],
        1 => [
          '#type' => 'textfield',
          '#default_value' => $payment_window_parts[1],
          '#size' => 2,
        ],
        2 => [
          '#type' => 'textfield',
          '#default_value' => $payment_window_parts[2],
          '#size' => 2,
        ],
      ],
    ];

    $options = [];
    foreach (UILocale::cases() as $case) {
      $options[$case->value] = $case->name;
    }

    $form['locale'] = [
      '#type' => 'select',
      '#title' => $this->t('Barion locale(language also)'),
      '#options' => $options,
      '#required' => TRUE,
      '#default_value' => $this->configuration['locale'],
    ];

    $form['reservation_period'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Reservation period'),
      '#description' => $this->t('The ReservationPeriod time of the payment start request in case of authorize-only payments. Should be between 1 minute and 1 year and in the format d.hh:mm:ss. Defaults to 30 minutes.'),
      '#default_value' => $this->configuration['reservation_period'],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedModes() {
    $barion_environment = new \ReflectionClass(BarionEnvironment::class);
    /** @var \Barion\Enumerations\BarionEnvironment[] $constants */
    $constants = $barion_environment->getConstants();
    $return = [];
    foreach ($constants as $constant) {
      $return[$constant->value] = $constant->name;
    }
    return $return;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);
    $values = $form_state->getValue(array_merge($form['#parents'], [
      'payment_window_label_item',
      'payment_window',
    ]));
    foreach ($values as $key => $value) {
      if (!is_numeric($value)) {
        $form_state->setErrorByName(
          implode('][', array_merge($form['#parents'], [
            'payment_window_label_item',
            'payment_window',
            $key,
          ])),
          $this->t('Value must be number')
        );
      }
      if (strlen($value) !== 2) {
        $form_state->setErrorByName(
          implode('][', array_merge($form['#parents'], [
            'payment_window_label_item',
            'payment_window',
            $key,
          ])),
          $this->t('Value must have exactly 2 characters, add zero in front if number is lower than 10')
        );
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    $values = $form_state->getValue($form['#parents']);
    $this->configuration['email'] = $values['email'];
    $this->configuration['private_key'] = $values['private_key'];
    $this->configuration['api_version'] = $values['api_version'];
    $this->configuration['payment_window'] = implode(':', $values['payment_window_label_item']['payment_window']);
    $this->configuration['locale'] = $values['locale'];
    $this->configuration['reservation_period'] = $values['reservation_period'];
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $payment_id = $order->getData('barion_payment_id');
    $payment_state = $this->getPaymentState($payment_id);
    $this->updatePaymentState($payment_id, $payment_state);
    switch ($payment_state->Status->value) {
      case 'Prepared':
        throw new PaymentGatewayException('Barion returned status prepared');

      case 'Started':
        throw new PaymentGatewayException('Barion payment not yet finished');

      case 'InProgress':
        throw new PaymentGatewayException('Barion returned status of in progress.');

      case 'Succeeded':
      case 'Reserved':
        break;

      case 'Canceled':
        throw new PaymentGatewayException('Barion returned status of canceled.');

      case 'Expired':
        throw new PaymentGatewayException('Payment process has expired.');

      case 'Authorized':
        throw new PaymentGatewayException('Barion returned status of payment authorized.');

      case 'PartiallySucceeded':
        throw new PaymentGatewayException('Barion returned status of payment partially succeeded.');

      case 'Waiting':
        throw new PaymentGatewayException('Barion returned status of payment waiting.');

      case 'Failed':
        throw new PaymentGatewayException('Barion returned status of failed.');
    }
    parent::onReturn($order, $request);
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request) {
    $payment_id = $request->query->get('paymentId', FALSE);
    $payment_state = $this->getPaymentState($payment_id);
    if ($payment_state->Status) {
      $this->updatePaymentState($payment_id, $payment_state);
    }
    parent::onNotify($request);
  }

  /**
   * Validates the payment state on Barion.
   *
   * @param mixed $payment_id
   *   Payment ID.
   *
   * @return \Barion\Models\Payment\PaymentStateResponseModel
   *   Payment state response.
   */
  public function getPaymentState($payment_id) {
    $barion_client = new BarionClient($this->configuration['private_key'], $this->configuration['api_version'], $this->getMode() === 'test' ? BarionEnvironment::Test : BarionEnvironment::Prod);
    return $barion_client->GetPaymentState($payment_id);
  }

  /**
   * Prepare the payment.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $order
   *   Order.
   * @param string $redirect_url
   *   Redirect URL.
   *
   * @return \Barion\Models\Payment\PreparePaymentResponseModel
   *   Prepare payment response.
   *
   * @throws \Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException
   *   Invalid plugin definition.
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   *   Plugin not found.
   */
  public function preparePayment(OrderInterface $order, $redirect_url) {
    $barion_client = new BarionClient($this->configuration['private_key'],
      $this->configuration['api_version'],
      $this->getMode() === 'test' ? BarionEnvironment::Test : BarionEnvironment::Prod);

    $items = $order->getItems();

    // Prepare payment transaction model.
    $transaction = new PaymentTransactionModel();
    $transaction->POSTransactionId = $order->id();
    $transaction->Payee = $this->configuration['email'];
    $total_paid_price = $order->getTotalPaid();
    $total_price = $order->getTotalPrice();
    if ($total_paid_price->greaterThan($total_price)) {
      throw new PaymentGatewayException('You have already completed payment, please contact administrator.');
    }
    $transaction->Total = $total_price->subtract($total_paid_price)
      ->getNumber();

    $variation_storage = $this->entityTypeManager
      ->getStorage('commerce_product_variation');

    // Add item models to transaction.
    foreach ($items as $order_item) {
      $item = new ItemModel();
      $item->Name = $order_item->getTitle();
      $item->Description = $order_item->getTitle();
      $item->Quantity = $order_item->getQuantity();
      $item->Unit = "piece";

      /** @var \Drupal\commerce_product\Entity\ProductVariation $variation */
      $variation = $variation_storage->load($order_item->getPurchasedEntityId());
      if ($variation) {
        $item->SKU = $variation->getSku();
      }
      else {
        $item->SKU = "none";
      }

      $item->UnitPrice = $order_item->getUnitPrice()->getNumber();
      $item->ItemTotal = $order_item->getTotalPrice()->getNumber();
      $transaction->AddItem($item);
    }
    // Prepare payment model.
    $prepare_payment = new PreparePaymentRequestModel();
    $prepare_payment->GuestCheckout = TRUE;
    $prepare_payment->FundingSources = [FundingSourceType::All];
    $prepare_payment->PaymentRequestId = $order->id();
    $prepare_payment->PayerHint = $order->getEmail();

    $currency_code = $total_price->getCurrencyCode();
    $prepare_payment->Currency = $this->assertCurrency($currency_code);
    // @todo Locale should probably be handled from site config.
    $prepare_payment->Locale = UILocale::from($this->configuration['locale']);

    $address = $order->get('billing_profile')->entity->get('address')
      ->getValue()[0];
    $billing_address = new BillingAddressModel();
    $billing_address->Country = $address['country_code'];
    $billing_address->City = $address['locality'];
    $billing_address->Zip = $address['postal_code'];
    $billing_address->Street = $address['address_line1'];
    $billing_address->Street2 = $address['address_line2'];
    $billing_address->Street3 = $address['address_line3'];

    $prepare_payment->OrderNumber = $order->getOrderNumber();
    $prepare_payment->BillingAddress = $billing_address;

    /** @var \Drupal\commerce_checkout\Entity\CheckoutFlowInterface $checkout_flow */
    $checkout_flow = $order->get('checkout_flow')->entity;
    $capture = $checkout_flow->get('configuration')['panes']['payment_process']['capture'];
    if ($capture) {
      $prepare_payment->PaymentType = PaymentType::Immediate;
    }
    else {
      $prepare_payment->PaymentType = PaymentType::Reservation;
      $prepare_payment->ReservationPeriod = $this->configuration['reservation_period'] ?: '0.00:30:00';
    }

    $prepare_payment->AddTransaction($transaction);
    $prepare_payment->PaymentWindow = $this->configuration['payment_window'];

    $prepare_payment->RedirectUrl = $redirect_url;
    $prepare_payment->CallbackUrl = $this->getNotifyUrl()->toString();

    $prepared_payment = $barion_client->PreparePayment($prepare_payment);
    return $prepared_payment;
  }

  /**
   * Asserts currency code.
   */
  public function assertCurrency($currency_code) {
    if (Currency::tryFrom($currency_code)) {
      return Currency::from($currency_code);
    }
    throw new PaymentGatewayException('Barion does not support currency of order');
  }

  /**
   * {@inheritdoc}
   */
  public function updatePaymentState($payment_id, $remote_state) {
    $payment_storage = $this->entityTypeManager
      ->getStorage('commerce_payment');
    $payment = $payment_storage->loadByProperties([
      'remote_id' => $payment_id,
      'payment_gateway' => $this->parentEntity->id(),
    ]);
    if (count($payment) === 1) {
      /** @var \Drupal\commerce_payment\Entity\Payment $payment */
      $payment = reset($payment);
      if ($state = $this->mapState($remote_state->Status->value)) {
        $payment->setState($state);
      }
      else {
        $this->logger->error("Could not map remote state on payment $payment_id");
      }
      $payment->setRemoteState($remote_state->Status->value);
      $payment->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment($order, $payment_state) {
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $payment = $payment_storage->create([
      'state' => $this->mapState($payment_state->Status->value),
      'amount' => $order->getTotalPrice(),
      'payment_gateway' => $this->parentEntity->id(),
      'order_id' => $order->id(),
      'remote_id' => $payment_state->PaymentId,
      'remote_state' => $payment_state->Status->value,
    ]);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function mapState($remote_state) {
    $state_map = [
      'Prepared' => 'new',
      'Started' => 'new',
      'InProgress' => 'authorization',
      'Reserved' => 'authorization',
      'Canceled' => 'authorization_voided',
      'Succeeded' => 'completed',
      'Failed' => 'authorization_voided',
      'Expired' => 'authorization_expired',
      'Authorized' => 'authorization',
      'PartiallySucceeded' => 'authorization',
      'Waiting' => 'new',
    ];
    return $state_map[$remote_state] ?? FALSE;
  }

  /**
   *
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment) {
    $remote_id = $payment->getRemoteId();
    if ($payment->getState()->value !== 'authorization') {
      throw new PaymentGatewayException($this->t('Only authorized payments can be cancelled.'));
    }
    try {
      $barion_client = new BarionClient($this->configuration['private_key'], $this->configuration['api_version'], $this->getMode() === 'test' ? BarionEnvironment::Test : BarionEnvironment::Prod);
      $model = new FinishReservationRequestModel($remote_id);
      $remote_payment = $barion_client->GetPaymentState($remote_id);
      $remote_transaction = reset($remote_payment->Transactions);
      $finish_transaction = new TransactionToFinishModel();
      $finish_transaction->Total = $remote_transaction->Total;
      $finish_transaction->TransactionId = $remote_transaction->TransactionId;
      $finish_transaction->Total = 0;
      $model->AddTransaction($finish_transaction);
      $barion_client->FinishReservation($model);

      $payment->setState('authorization_voided');
      $payment->setRemoteState('Canceled');
      $payment->save();
    }
    catch (\Exception $e) {
      $this->logger->warning($e->getMessage());
      throw new PaymentGatewayException($this->t('We could not void your payment with Barion. Please try again or contact us if the problem persists.'));
    }
  }

  /**
   *
   * {@inheritdoc}
   */
  public function capturePayment(PaymentInterface $payment, Price $amount = NULL) {
    $remote_id = $payment->getRemoteId();
    if ($payment->getState()->value !== 'authorization') {
      throw new PaymentGatewayException($this->t('Only authorized payments can be captured.'));
    }
    try {
      $barion_client = new BarionClient($this->configuration['private_key'], $this->configuration['api_version'], $this->getMode() === 'test' ? BarionEnvironment::Test : BarionEnvironment::Prod);
      $model = new FinishReservationRequestModel($remote_id);
      $remote_payment = $barion_client->GetPaymentState($remote_id);
      $remote_transaction = reset($remote_payment->Transactions);

      $finish_transaction = new TransactionToFinishModel();
      $finish_transaction->Total = $remote_transaction->Total;
      $finish_transaction->TransactionId = $remote_transaction->TransactionId;

      if ($amount) {
        if ($amount->getNumber() > $finish_transaction->Total) {
          throw new PaymentGatewayException($this->t('Cannot capture more than the original payment amount.'));
        }
        $finish_transaction->Total = $amount->getNumber();
        $payment->setAmount($amount);
      }
      $model->AddTransaction($finish_transaction);
      $barion_client->FinishReservation($model);
      $payment->setState('completed');
      $payment->setRemoteState('Succeeded');
      $payment->save();
    }
    catch (\Exception $e) {
      $this->logger->warning($e->getMessage());
      throw new PaymentGatewayException($this->t('We could not capture your payment with Barion. Please try again or contact us if the problem persists.'));
    }
  }

}
