/**
 * @file
 * Defines behaviors for the CardPointe hosted iframe payment method form.
 */
((Drupal, once) => {
  /**
   * The Drupal Commerce CardPointe Element Instances.
   *
   * Only one instance is supported.
   *
   * @type {Map}
   */
  Drupal.CommerceCardPointeInstances = new Map();

  class CardPointeHostedIframe {
    /**
     * Whether the component has been initialized.
     */
    initialized;

    /**
     * Whether the token has been resolved.
     */
    resolved;

    /**
     * The settings.
     */
    settings;

    /**
     * The form.
     */
    form;

    /**
     * The token.
     */
    token;

    /**
     * The token field.
     */
    tokenField;

    /**
     * The expiration field.
     */
    expirationField;

    /**
     * The "submit" button.
     */
    submitButton;

    /**
     * Bound reference to messageHandler for proper cleanup.
     */
    boundMessageHandler;

    /**
     * Bound reference to oneShotMessageHandler for proper cleanup.
     */
    boundOneShotHandler;

    /**
     * Bound reference to handleSubmit for proper cleanup.
     */
    boundHandleSubmit;

    /**
     * Timeout ID for cleanup.
     */
    timeoutId;

    /**
     * The Commerce CardPointe constructor.
     */
    constructor() {
      if (!this.initialized) {
        this.initialized = true;
        this.resolved = false;
        this.settings = null;
        this.form = null;
        this.token = null;
        this.tokenField = null;
        this.expirationField = null;
        this.submitButton = null;
        // Store bound function references to properly remove listeners
        this.boundMessageHandler = null;
        this.boundOneShotHandler = null;
        this.boundHandleSubmit = null;
        this.timeoutId = null;
      }
    }

    /**
     * Initialize the CardPointe component.
     *
     * @param {Object} settings
     *   The settings.
     */
    async initialize(settings) {
      this.settings = settings;
      this.form = document.querySelector('form.commerce-checkout-flow');
      this.tokenField = document.getElementById('token');
      this.expirationField = document.getElementById('expiration');
      this.submitButton = this.form.querySelector('.button--primary');

      // Store the bound references for proper cleanup
      this.boundMessageHandler = this.messageHandler.bind(this);
      this.boundHandleSubmit = this.handleSubmit.bind(this);
      this.boundOneShotHandler = this.oneShotMessageHandler.bind(this);

      window.addEventListener('message', this.boundMessageHandler, false);
      this.submitButton.addEventListener('click', this.boundHandleSubmit);
    }

    tokenIsValid() {
      return this.token && this.token.message && this.token.expiry;
    }

    processData(rawData) {
      try {
        const data =
          typeof rawData === 'string' ? JSON.parse(rawData) : rawData;
        if (!data || typeof data.message !== 'string' || !data.expiry) {
          return null;
        }

        this.token = data;
        this.tokenField.value = this.token.message;
        this.expirationField.value = this.token.expiry;
        return data;
      } catch (e) {
        // Ignore malformed data.
      }
      return null;
    }

    /**
     * Validates the message origin.
     *
     * @param {string} origin
     *   The origin to validate.
     * @return {boolean}
     *   True if valid, false otherwise.
     */
    isValidOrigin(origin) {
      try {
        const tokenUrl = new URL(this.settings.tokenFrameUrl);
        const eventOrigin = new URL(origin);
        return tokenUrl.origin === eventOrigin.origin;
      } catch (e) {
        return false;
      }
    }

    // Global message handler keeps the token up to date.
    messageHandler(event) {
      if (!this.isValidOrigin(event.origin)) {
        return;
      }
      this.processData(event.data);
    }

    oneShotMessageHandler(oneShotEvent) {
      if (this.resolved) {
        return;
      }
      if (!this.isValidOrigin(oneShotEvent.origin)) {
        return;
      }
      try {
        this.processData(oneShotEvent.data);
        if (this.tokenIsValid()) {
          this.resolved = true;
          this.cleanup();
          this.submitButton.disabled = false;
          this.form.submit();
        }
      } catch (e) {
        // Ignore malformed data and keep waiting.
      }
    }

    /**
     * Clean up event listeners and timeout.
     */
    cleanup() {
      if (this.timeoutId) {
        clearTimeout(this.timeoutId);
        this.timeoutId = null;
      }
      window.removeEventListener('message', this.boundOneShotHandler, false);
    }

    handleSubmit(event) {
      // If we already have a token, just let the form submit.
      if (this.tokenIsValid()) {
        return;
      }

      event.preventDefault();

      this.submitButton.disabled = true;
      this.resolved = false;

      window.addEventListener('message', this.boundOneShotHandler, false);

      // Optional hard timeout so the UI isn’t stuck forever.
      this.timeoutId = setTimeout(this.handleTimeout.bind(this), 5000);
    }

    handleTimeout() {
      if (this.resolved) {
        return;
      }
      this.resolved = true;
      this.cleanup();
      this.submitButton.disabled = false;
      const messages = new Drupal.Message();
      messages.add(
        Drupal.t('We could not get a payment token. Please try again.'),
        { type: 'error' },
      );
    }
  }

  Drupal.behaviors.hostedIframe = {
    async attach(context, drupalSettings) {
      // Validate all required dependencies.
      if (!Drupal || !drupalSettings) {
        console.error('Required dependencies are not available');
        return;
      }

      const settings = drupalSettings.commerceCardpointe;
      if (!settings?.tokenFrameUrl) {
        console.error('Required settings are missing');
        return;
      }

      const tokenField = document.getElementById('token');
      const expirationField = document.getElementById('expiration');

      if (!tokenField || !expirationField) {
        return;
      }

      const [form] = once(
        'cardpointe-hostediframe-processed',
        document.querySelector('form.commerce-checkout-flow'),
        context,
      );
      if (form) {
        const cardpointeComponent = new CardPointeHostedIframe();
        await cardpointeComponent.initialize(settings);
        Drupal.CommerceCardPointeInstances.set(form, cardpointeComponent);
      }
    },
  };
})(Drupal, once);
