<?php

namespace Drupal\commerce_cashpresso;

use Drupal\commerce_price\Calculator;
use Drupal\commerce_price\Price;

/**
 * Provides a value object for storing data from the Get Partner Info request.
 */
final class PartnerInfo {

  /**
   * The pending status of partner accounts.
   *
   * @var string
   */
  const STATUS_PENDING = 'PENDING';

  /**
   * The active status of partner accounts.
   *
   * @var string
   */
  const STATUS_ACTIVE = 'ACTIVE';

  /**
   * The declined status of partner accounts.
   *
   * @var string
   */
  const STATUS_DECLINED = 'DECLINED';

  /**
   * The partner's brand name.
   *
   * The shop or otherwise customer facing name. cashpresso will use this name
   * in all communication towards your customers.
   *
   * @var string
   */
  protected string $brandName;

  /**
   * The legal company name.
   *
   * @var string
   */
  protected string $companyName;

  /**
   * The company website url.
   *
   * @var string
   */
  protected string $companyUrl;

  /**
   * The partner's email address.
   *
   * @var string
   */
  protected string $email;

  /**
   * The holder's name.
   *
   * @var string
   */
  protected string $holder;

  /**
   * The IBAN.
   *
   * @var string
   */
  protected string $iban;

  /**
   * Whether the partner is allowed to set interest per purchase.
   *
   * @var bool
   */
  protected bool $interestFreeEnabled;

  /**
   * The maximum number of interest free days the partner is allowed to offer.
   *
   * @var int
   */
  protected int $interestFreeMaxDuration;

  /**
   * The status of the partner account. One of PENDING, ACTIVE or DECLINED.
   *
   * Please note that the partner can only send in payment requests if the
   * account is in state ACTIVE.
   *
   * @var string
   */
  protected string $status;

  /**
   * Three-letter ISO currency code, e.g. EUR.
   *
   * @var string
   */
  protected string $currencyCode;

  /**
   * The minimum payback amount.
   *
   * The minimum amount a customer has to pay per month in the respective
   * currency value. cashpresso payback terms usually come with a paybackRate
   * in percent and a minimum amount.
   *
   * @var float
   */
  protected float $minPaybackAmount;

  /**
   * The payback rate in percent.
   *
   * You can always calculate the first instalment of a product with price P
   * using this formula: min(P, max(minPaybackAmount, P * 0.01 * paybackRate)).
   *
   * If you have a product with price 750€, paybackRate is 3.00 and
   * minPaybackAmount is 20€, the first instalment is:
   *   min(750, max(20, 750*0.01*3.00)) = 22.5
   *
   * If you have a product with price 500€ and the same terms, the first
   * instalment is:
   *   min(500, max(20, 500*0.01*3.00)) = 20
   *
   * @var float
   */
  protected float $paybackRate;

  /**
   * The highest amount new customers can finance with cashpresso.
   *
   * @var \Drupal\commerce_price\Price|null
   *   The financing limit.
   */
  protected ?Price $financingLimit = NULL;

  /**
   * The highest amount for prepayment with cashpresso.
   *
   * @var \Drupal\commerce_price\Price|null
   *   The prepayment limit.
   */
  protected ?Price $prepaymentLimit = NULL;

  /**
   * The total limit.
   *
   * @var \Drupal\commerce_price\Price|null
   *   The total limit.
   */
  protected ?Price $totalLimit = NULL;

  /**
   * The minimum nominal interest rate in percent.
   *
   * @var float|null
   */
  protected ?float $minNominalInterestRate;

  /**
   * The maximum nominal interest rate in percent.
   *
   * @var float|null
   */
  protected ?float $maxNominalInterestRate;

  /**
   * The minimum effective interest rate in percent.
   *
   * @var float|null
   */
  protected ?float $minEffectiveInterestRate;

  /**
   * The maximum effective interest rate in percent.
   *
   * @var float|null
   */
  protected ?float $maxEffectiveInterestRate;

  /**
   * Interest free days granted by cashpresso.
   *
   * The number of interest free days cashpresso grants per purchase. Any amount
   * of interest free days the partner grants will be added on top of this.
   *
   * @var int
   */
  protected int $interestFreeCashpresso;

  /**
   * Factory method allowing the object to be instantiated by an array.
   *
   * @param array $values
   *   The values, as returned by Get Partner Info webservice.
   *
   * @return static
   *   A new PartnerInfo object.
   */
  public static function fromArray(array $values) {
    $instance = new PartnerInfo();
    $instance->setBrandName($values['brandName'] ?? '');
    $instance->setCompanyName($values['companyName'] ?? '');
    $instance->setCompanyUrl($values['companyUrl'] ?? '');
    $instance->setEmail($values['email'] ?? '');
    $instance->setHolder($values['holder'] ?? '');
    $instance->setIban($values['iban'] ?? '');
    $instance->setInterestFreeEnabled((bool) $values['interestFreeEnabled'] ?? FALSE);
    $instance->setInterestFreeMaxDuration((int) $values['interestFreeMaxDuration'] ?? 0);
    $instance->setStatus($values['status'] ?? self::STATUS_PENDING);
    $instance->setCurrencyCode($values['currency'] ?: 'EUR');
    $instance->setMinPaybackAmount((float) $values['minPaybackAmount']);
    $instance->setPaybackRate((float) $values['paybackRate']);
    if (!empty($values['limit']['financing'])) {
      $instance->setFinancingLimit(new Price((string) $values['limit']['financing'], $values['currency']));
    }
    if (!empty($values['limit']['prepayment'])) {
      $instance->setPrepaymentLimit(new Price((string) $values['limit']['prepayment'], $values['currency']));
    }
    if (!empty($values['limit']['total'])) {
      $instance->setTotalLimit(new Price((string) $values['limit']['total'], $values['currency']));
    }
    $instance->setMinNominalInterestRate($values['interest']['nominal']['min'] ?? NULL);
    $instance->setMaxNominalInterestRate($values['interest']['nominal']['max'] ?? NULL);
    $instance->setMinEffectiveInterestRate($values['interest']['effective']['min'] ?? NULL);
    $instance->setMaxEffectiveInterestRate($values['interest']['effective']['max'] ?? NULL);
    $instance->setInterestFreeCashpresso((int) $values['interestFreeCashpresso']);
    return $instance;
  }

  /**
   * Return the brand name.
   *
   * @return string
   *   The brand name.
   */
  public function getBrandName(): string {
    return $this->brandName;
  }

  /**
   * Set the brand name.
   *
   * @param string $brand_name
   *   The brand name.
   *
   * @return $this
   */
  public function setBrandName(string $brand_name) {
    $this->brandName = $brand_name;
    return $this;
  }

  /**
   * Get the company name.
   *
   * @return string
   *   The company name.
   */
  public function getCompanyName(): string {
    return $this->companyName;
  }

  /**
   * Set the company name.
   *
   * @param string $company_name
   *   The company name.
   *
   * @return $this
   */
  public function setCompanyName(string $company_name) {
    $this->companyName = $company_name;
    return $this;
  }

  /**
   * Get the company url.
   *
   * @return string
   *   The company url.
   */
  public function getCompanyUrl(): string {
    return $this->companyUrl;
  }

  /**
   * Set the company url.
   *
   * @param string $company_url
   *   The company url.
   *
   * @return $this
   */
  public function setCompanyUrl(string $company_url) {
    $this->companyUrl = $company_url;
    return $this;
  }

  /**
   * Get the email address.
   *
   * @return string
   *   The email address.
   */
  public function getEmail(): string {
    return $this->email;
  }

  /**
   * Set the email address.
   *
   * @param string $email
   *   The email address.
   *
   * @return $this
   */
  public function setEmail(string $email) {
    $this->email = $email;
    return $this;
  }

  /**
   * Gets the holder.
   *
   * @return string
   *   The holder.
   */
  public function getHolder(): string {
    return $this->holder;
  }

  /**
   * Sets the holder.
   *
   * @param string $holder
   *   The holder.
   *
   * @return $this
   */
  public function setHolder(string $holder) {
    $this->holder = $holder;
    return $this;
  }

  /**
   * Gets the IBAN.
   *
   * @return string
   *   The IBAN.
   */
  public function getIban(): string {
    return $this->iban;
  }

  /**
   * Sets the IBAN.
   *
   * @param string $iban
   *   The IBAN.
   *
   * @return $this
   */
  public function setIban(string $iban) {
    $this->iban = $iban;
    return $this;
  }

  /**
   * Gets whether interest free is enabled.
   *
   * @return bool
   *   TRUE, if interest free is enabled. FALSE otherwise.
   */
  public function isInterestFreeEnabled(): bool {
    return $this->interestFreeEnabled;
  }

  /**
   * Sets whether interest free is enabled.
   *
   * @param bool $interestFreeEnabled
   *   TRUE, if interest free should be enabled. FALSE otherwise.
   *
   * @return $this
   */
  public function setInterestFreeEnabled(bool $interestFreeEnabled) {
    $this->interestFreeEnabled = $interestFreeEnabled;
    return $this;
  }

  /**
   * Returns the max interest free duration.
   *
   * @return int
   *   The max interest free duration.
   */
  public function getInterestFreeMaxDuration(): int {
    return $this->interestFreeMaxDuration;
  }

  /**
   * Sets the max interest free duration.
   *
   * @param int $interest_free_max_duration
   *   The max interest free duration.
   *
   * @return $this
   */
  public function setInterestFreeMaxDuration(int $interest_free_max_duration) {
    $this->interestFreeMaxDuration = $interest_free_max_duration;
    return $this;
  }

  /**
   * Gets the partner account status.
   *
   * @return string
   *   The partner account status.
   */
  public function getStatus(): string {
    return $this->status;
  }

  /**
   * Sets the partner account status.
   *
   * @param string $status
   *   The partner account status.
   *
   * @return $this
   */
  public function setStatus(string $status) {
    $this->status = $status;
    return $this;
  }

  /**
   * Gets the currency code.
   *
   * @return string
   *   The currency code.
   */
  public function getCurrencyCode(): string {
    return $this->currencyCode;
  }

  /**
   * Sets the currency code.
   *
   * @param string $currency_code
   *   The currency code.
   *
   * @return $this
   */
  public function setCurrencyCode(string $currency_code) {
    $this->currencyCode = $currency_code;
    return $this;
  }

  /**
   * Gets the minimum payback amount.
   *
   * @return float
   *   The minimum payback amount.
   */
  public function getMinPaybackAmount(): float {
    return $this->minPaybackAmount;
  }

  /**
   * Sets the minimum payback amount.
   *
   * @param float $minPaybackAmount
   *   The minimum payback amount.
   *
   * @return $this
   */
  public function setMinPaybackAmount(float $minPaybackAmount) {
    $this->minPaybackAmount = $minPaybackAmount;
    return $this;
  }

  /**
   * Gets the payback rate.
   *
   * @return float
   *   The payback rate.
   */
  public function getPaybackRate(): float {
    return $this->paybackRate;
  }

  /**
   * Sets the payback rate.
   *
   * @param float $paybackRate
   *   The payback rate.
   *
   * @return $this
   */
  public function setPaybackRate(float $paybackRate) {
    $this->paybackRate = $paybackRate;
    return $this;
  }

  /**
   * Gets the financing limit.
   *
   * @return \Drupal\commerce_price\Price|null
   *   The financing limit.
   */
  public function getFinancingLimit(): ?Price {
    return $this->financingLimit;
  }

  /**
   * Sets the financing limit.
   *
   * @param \Drupal\commerce_price\Price $financing_limit
   *   The financing limit.
   *
   * @return $this
   */
  public function setFinancingLimit(Price $financing_limit) {
    $this->financingLimit = $financing_limit;
    return $this;
  }

  /**
   * Sets the prepayment limit.
   *
   * @return \Drupal\commerce_price\Price|null
   *   The prepayment limit.
   */
  public function getPrepaymentLimit(): ?Price {
    return $this->prepaymentLimit;
  }

  /**
   * Sets the prepayment limit.
   *
   * @param \Drupal\commerce_price\Price $prepayment_limit
   *   The prepayment limit.
   *
   * @return $this
   */
  public function setPrepaymentLimit(Price $prepayment_limit) {
    $this->prepaymentLimit = $prepayment_limit;
    return $this;
  }

  /**
   * Gets the total limit.
   *
   * @return \Drupal\commerce_price\Price|null
   *   The total limit.
   */
  public function getTotalLimit(): ?Price {
    return $this->totalLimit;
  }

  /**
   * Sets the total limit.
   *
   * @param \Drupal\commerce_price\Price $total_limit
   *   The total limit.
   *
   * @return $this
   */
  public function setTotalLimit(Price $total_limit) {
    $this->totalLimit = $total_limit;
    return $this;
  }

  /**
   * Gets the minimum nominal interest rate.
   *
   * @return float|null
   *   The minimum nominal interest rate.
   */
  public function getMinNominalInterestRate(): ?float {
    return $this->minNominalInterestRate;
  }

  /**
   * Sets the minimum nominal interest rate.
   *
   * @param float|null $rate
   *   The minimum nominal interest rate.
   *
   * @return $this
   */
  public function setMinNominalInterestRate(?float $rate) {
    $this->minNominalInterestRate = $rate;
    return $this;
  }

  /**
   * Gets the maximum nominal interest rate.
   *
   * @return float|null
   *   The maximum nominal interest rate.
   */
  public function getMaxNominalInterestRate(): ?float {
    return $this->maxNominalInterestRate;
  }

  /**
   * Sets the maximum nominal interest rate.
   *
   * @param float|null $rate
   *   The maximum nominal interest rate.
   *
   * @return $this
   */
  public function setMaxNominalInterestRate(?float $rate) {
    $this->maxNominalInterestRate = $rate;
    return $this;
  }

  /**
   * Gets the minimum effective interest rate.
   *
   * @return float|null
   *   The minimum effective interest rate.
   */
  public function getMinEffectiveInterestRate(): ?float {
    return $this->minEffectiveInterestRate;
  }

  /**
   * Sets the minimum effective interest rate.
   *
   * @param float|null $rate
   *   The minimum effective interest rate.
   *
   * @return $this
   */
  public function setMinEffectiveInterestRate(?float $rate) {
    $this->minEffectiveInterestRate = $rate;
    return $this;
  }

  /**
   * Gets the maximum effective interest rate.
   *
   * @return float|null
   *   The maximum effective interest rate.
   */
  public function getMaxEffectiveInterestRate(): ?float {
    return $this->maxEffectiveInterestRate;
  }

  /**
   * Sets the maximum effective interest rate.
   *
   * @param float|null $rate
   *   The maximum effective interest rate.
   *
   * @return $this
   */
  public function setMaxEffectiveInterestRate(?float $rate) {
    $this->maxEffectiveInterestRate = $rate;
    return $this;
  }

  /**
   * Gets the number of interest free days cashpresso grants per purchase.
   *
   * @return int
   *   The number of interest free days cashpresso grants per purchase.
   */
  public function getInterestFreeCashpresso(): int {
    return $this->interestFreeCashpresso;
  }

  /**
   * Sets the number of interest free days cashpresso grants per purchase.
   *
   * @param int $interest_free_cashpresso
   *   The number of interest free days cashpresso grants per purchase.
   *
   * @return $this
   */
  public function setInterestFreeCashpresso(int $interest_free_cashpresso) {
    $this->interestFreeCashpresso = $interest_free_cashpresso;
    return $this;
  }

  /**
   * Calculates the instalment price for the given product price.
   *
   * The formula for product price P is:
   *   min(P, max(minPaybackAmount, P * 0.01 * paybackRate))
   *
   * @param \Drupal\commerce_price\Price $product_price
   *   The product price.
   *
   * @return \Drupal\commerce_price\Price|null
   *   The instalment price, or NULL, if the partner info is incomplete (which
   *   should never really be possible in fact).
   */
  public function calculateInstalmentPrice(Price $product_price): ?Price {
    $min_payback_amount = (string) $this->getMinPaybackAmount();
    $payback_rate = (string) $this->getPaybackRate();
    if (empty($min_payback_amount) || empty($payback_rate)) {
      return NULL;
    }
    $price_amount = $product_price->getNumber();
    // Normalize percentage number to a rate decimal string.
    $payback_rate = Calculator::multiply($payback_rate, '0.01');
    // Calculate the payback amount for the given price.
    $payback_amount = Calculator::multiply($price_amount, $payback_rate);
    // Ensure the minimum payback amount.
    if (Calculator::compare($min_payback_amount, $payback_amount) > 0) {
      $payback_amount = $min_payback_amount;
    }
    // Ensure that the payback amount is not higher than the product price.
    if (Calculator::compare($payback_amount, $price_amount) > 0) {
      $payback_amount = $price_amount;
    }
    return new Price($payback_amount, $product_price->getCurrencyCode());
  }

}
