<?php

namespace Drupal\Tests\commerce_ccavenue\Functional;

use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_payment\Entity\PaymentGateway;
use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\Tests\commerce\Functional\CommerceBrowserTestBase;

/**
 * Tests the CCAvenue payment gateway.
 *
 * @group commerce_ccavenue
 */
class CCAvenuePaymentTest extends CommerceBrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'commerce_ccavenue',
    'commerce_cart',
    'commerce_checkout',
    'commerce_payment',
    'commerce_payment_example',
  ];

  /**
   * The payment gateway.
   *
   * @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface
   */
  protected $gateway;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a payment gateway.
    $this->gateway = PaymentGateway::create([
      'id' => 'ccavenue_test',
      'label' => 'CCAvenue Test',
      'plugin' => 'ccavenue_redirect',
      'configuration' => [
        'merchant_id' => 'test_merchant_123',
        'access_code' => 'test_access_code',
        'working_key' => 'test_working_key_12345',
        'currency' => 'INR',
        'language' => 'EN',
        'mode' => 'test',
      ],
    ]);
    $this->gateway->save();
  }

  /**
   * Tests the payment gateway configuration form.
   */
  public function testPaymentGatewayConfiguration() {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/commerce/config/payment-gateways');
    $this->assertSession()->pageTextContains('CCAvenue Test');

    // Test the configuration form.
    $this->drupalGet('admin/commerce/config/payment-gateways/manage/ccavenue_test');
    $this->assertSession()->fieldExists('configuration[ccavenue_redirect][merchant_id]');
    $this->assertSession()->fieldExists('configuration[ccavenue_redirect][access_code]');
    $this->assertSession()->fieldExists('configuration[ccavenue_redirect][working_key]');
    $this->assertSession()->fieldExists('configuration[ccavenue_redirect][currency]');

    // Update configuration.
    $edit = [
      'configuration[ccavenue_redirect][merchant_id]' => 'updated_merchant_123',
      'configuration[ccavenue_redirect][access_code]' => 'updated_access_code',
      'configuration[ccavenue_redirect][working_key]' => 'updated_working_key',
      'configuration[ccavenue_redirect][currency]' => 'USD',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('Saved the CCAvenue Test payment gateway.');
  }

  /**
   * Tests the checkout process with CCAvenue.
   */
  public function testCheckoutWithCCAvenue() {
    // Create a product.
    $variation = ProductVariation::create([
      'type' => 'default',
      'sku' => 'TEST_SKU',
      'title' => 'Test Product Variation',
      'price' => [
        'number' => '10.00',
        'currency_code' => 'INR',
      ],
    ]);
    $variation->save();

    $product = Product::create([
      'type' => 'default',
      'title' => 'Test Product',
      'variations' => [$variation],
      'stores' => [$this->store],
    ]);
    $product->save();

    // Add product to cart.
    $this->drupalLogin($this->createUser());
    $this->drupalGet($product->toUrl());
    $this->submitForm([], 'Add to cart');

    // Go to checkout.
    $this->drupalGet('checkout');
    $this->assertSession()->pageTextContains('Order Summary');

    // Continue to payment.
    $this->submitForm([], 'Continue to review');
    $this->assertSession()->pageTextContains('Payment information');
    $this->assertSession()->pageTextContains('CCAvenue Redirect');

    // Select CCAvenue as payment method.
    $this->submitForm(['payment_information[payment_method]' => 'new--credit_card--ccavenue_test'], 'Continue to review');
    $this->assertSession()->pageTextContains('Review');

    // Complete the order (this would normally redirect to CCAvenue).
    $this->submitForm([], 'Pay and complete purchase');

    // In a real test, we would mock the CCAvenue response and test the return handling.
  }

  /**
   * Tests the payment gateway plugin configuration.
   */
  public function testPaymentGatewayPlugin() {
    $plugin = $this->gateway->getPlugin();

    $this->assertEquals('ccavenue_redirect', $plugin->getPluginId());
    $this->assertEquals('CCAvenue Redirect', $plugin->getDisplayLabel());
    $this->assertEquals('test_merchant_123', $plugin->getConfiguration()['merchant_id']);
    $this->assertEquals('test_access_code', $plugin->getConfiguration()['access_code']);
    $this->assertEquals('test_working_key_12345', $plugin->getConfiguration()['working_key']);
    $this->assertEquals('INR', $plugin->getConfiguration()['currency']);
  }

}
