<?php

declare(strict_types=1);

namespace Drupal\commerce_checkout_lock;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\Core\Lock\LockBackendInterface;
use Psr\Log\LoggerInterface;

/**
 * Manages checkout locks to prevent concurrent order submissions.
 */
class CheckoutLockManager implements CheckoutLockManagerInterface {

  private const LOCK_TIMEOUT = 60;
  private const LOCK_PREFIX = 'commerce_checkout';

  public function __construct(
    protected LockBackendInterface $lockBackend,
    protected LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function acquire(OrderInterface $order): bool {
    $name = $this->getLockName($order);
    $acquired = $this->lockBackend->acquire($name, self::LOCK_TIMEOUT);

    $this->logger->debug($acquired ? 'Lock acquired: @name' : 'Lock failed: @name', [
      '@name' => $name,
    ]);

    return $acquired;
  }

  /**
   * {@inheritdoc}
   */
  public function release(OrderInterface $order): void {
    $name = $this->getLockName($order);
    $this->lockBackend->release($name);
    $this->logger->debug('Lock released: @name', ['@name' => $name]);
  }

  /**
   * Generates the lock name for an order.
   */
  private function getLockName(OrderInterface $order): string {
    return self::LOCK_PREFIX . ':' . $order->id();
  }

}
