<?php

declare(strict_types=1);

namespace Drupal\commerce_checkout_lock\EventSubscriber;

use Drupal\commerce_checkout\Event\CheckoutEvents;
use Drupal\commerce_checkout_lock\CheckoutLockManagerInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Event\OrderEvent;
use Psr\Log\LoggerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Handles checkout lock lifecycle and cleanup.
 */
class CheckoutLockSubscriber implements EventSubscriberInterface {

  /**
   * Orders tracked for cleanup.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface[]
   */
  private array $tracked = [];

  public function __construct(
    protected CheckoutLockManagerInterface $lockManager,
    protected LoggerInterface $logger,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      CheckoutEvents::COMPLETION => ['onCompletion', -100],
      KernelEvents::TERMINATE => ['onTerminate', -100],
    ];
  }

  /**
   * Registers a lock for safety-net cleanup.
   */
  public function track(OrderInterface $order): void {
    $this->tracked[$order->id()] = $order;
  }

  /**
   * Removes an order from tracking.
   */
  public function untrack(OrderInterface $order): void {
    unset($this->tracked[$order->id()]);
  }

  /**
   * Releases lock on successful checkout completion.
   */
  public function onCompletion(OrderEvent $event): void {
    $order = $event->getOrder();
    $this->lockManager->release($order);
    $this->untrack($order);
  }

  /**
   * Safety net: releases orphaned locks on request termination.
   */
  public function onTerminate(TerminateEvent $event): void {
    foreach ($this->tracked as $id => $order) {
      $this->logger->notice('Releasing orphaned lock for order @id', ['@id' => $id]);
      $this->lockManager->release($order);
    }
    $this->tracked = [];
  }

}
