<?php

declare(strict_types=1);

namespace Drupal\commerce_checkout_lock\Plugin\Commerce\CheckoutPane;

use Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowInterface;
use Drupal\commerce_checkout\Plugin\Commerce\CheckoutPane\CheckoutPaneBase;
use Drupal\commerce_checkout_lock\CheckoutLockManagerInterface;
use Drupal\commerce_checkout_lock\EventSubscriber\CheckoutLockSubscriber;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Acquires a lock to prevent duplicate checkout submissions.
 *
 * @CommerceCheckoutPane(
 *   id = "checkout_lock",
 *   label = @Translation("Checkout Lock"),
 *   default_step = "_disabled",
 * )
 */
class CheckoutLock extends CheckoutPaneBase {

  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    CheckoutFlowInterface $checkout_flow,
    EntityTypeManagerInterface $entity_type_manager,
    protected CheckoutLockManagerInterface $lockManager,
    protected CheckoutLockSubscriber $lockSubscriber,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $checkout_flow, $entity_type_manager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, ?CheckoutFlowInterface $checkout_flow = NULL) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $checkout_flow,
      $container->get('entity_type.manager'),
      $container->get('commerce_checkout_lock.lock_manager'),
      $container->get('commerce_checkout_lock.event_subscriber'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildPaneForm(array $pane_form, FormStateInterface $form_state, array &$complete_form): array {
    $complete_form['#attached']['library'][] = 'commerce_checkout_lock/checkout_protection';
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function submitPaneForm(array &$pane_form, FormStateInterface $form_state, array &$complete_form): void {
    if (!$this->lockManager->acquire($this->order)) {
      $this->messenger()->addError($this->t('This order is being processed. Please wait.'));
      $this->checkoutFlow->redirectToStep($this->getStepId());
      return;
    }
    $this->lockSubscriber->track($this->order);
  }

}
