<?php

namespace Drupal\commerce_configurable_order_total\Plugin\Field\FieldFormatter;

use Drupal\commerce_order\OrderTotalSummaryInterface;
use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'configurable order total summary' formatter.
 */
#[FieldFormatter(
  id: 'commerce_configurable_order_total_summary',
  label: new TranslatableMarkup('Commerce configurable order total summary'),
  field_types: [
    'commerce_price',
  ],
)]
class ConfigurableOrderTotalSummary extends FormatterBase implements ContainerFactoryPluginInterface {

  /**
   * The order total summary service.
   *
   * @var \Drupal\commerce_order\OrderTotalSummaryInterface
   */
  protected $orderTotalSummary;

  /**
   * Constructs a ConfigurableOrderTotalSummary object.
   *
   * @param string $plugin_id
   *   The plugin ID for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\commerce_order\OrderTotalSummaryInterface $order_total_summary
   *   Builds the totals for the given order.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, $label, $view_mode, array $third_party_settings, OrderTotalSummaryInterface $order_total_summary) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $label, $view_mode, $third_party_settings);

    $this->orderTotalSummary = $order_total_summary;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('commerce_order.order_total_summary')
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return [
      'disable_subtotal' => FALSE,
      'disable_adjustments' => FALSE,
      'disable_totals' => FALSE,
      'strip_trailing_zeroes' => FALSE,
    ] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    $elements = parent::settingsForm($form, $form_state);

    $elements['disable_subtotal'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable subtotal'),
      '#default_value' => $this->getSetting('disable_subtotal'),
    ];
    $elements['disable_adjustments'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable adjustments'),
      '#default_value' => $this->getSetting('disable_adjustments'),
    ];
    $elements['disable_totals'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable totals'),
      '#default_value' => $this->getSetting('disable_totals'),
    ];
    $elements['strip_trailing_zeroes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Strip trailing zeroes after the decimal point'),
      '#default_value' => $this->getSetting('strip_trailing_zeroes'),
    ];

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $items->getEntity();
    $settings = $this->getSettings();
    $elements = [];
    if (!$items->isEmpty()) {
      $elements[0] = [
        '#theme' => 'commerce_configurable_order_total_summary',
        '#order_entity' => $order,
        '#totals' => $this->orderTotalSummary->buildTotals($order),
        '#disable_subtotal' => $settings['disable_subtotal'],
        '#disable_adjustments' => $settings['disable_adjustments'],
        '#disable_totals' => $settings['disable_totals'],
        '#strip_trailing_zeroes' => $settings['strip_trailing_zeroes'],
      ];
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function isApplicable(FieldDefinitionInterface $field_definition) {
    $entity_type = $field_definition->getTargetEntityTypeId();
    $field_name = $field_definition->getName();
    return $entity_type == 'commerce_order' && $field_name == 'total_price';
  }

}
