<?php

namespace Drupal\commerce_configurable_order_total\Plugin\views\area;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsArea;
use Drupal\views\Plugin\views\area\AreaPluginBase;
use Drupal\views\Plugin\views\argument\NumericArgument;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines a configurable order total area handler.
 *
 * Shows the order total field in the footer of a View.
 *
 * @ingroup views_area_handlers
 */
#[ViewsArea("commerce_configurable_order_total")]
class ConfigurableOrderTotalArea extends AreaPluginBase {

  /**
   * Order Storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $orderStorage;

  /**
   * Constructs a new ConfigurableOrderTotalArea instance.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   Entity type manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->orderStorage = $entity_type_manager->getStorage('commerce_order');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['disable_subtotal'] = ['default' => FALSE];
    $options['disable_adjustments'] = ['default' => FALSE];
    $options['disable_totals'] = ['default' => FALSE];
    $options['strip_trailing_zeroes'] = ['default' => FALSE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $form['empty']['#description'] = $this->t("Even if selected, this area handler will never render if a valid order cannot be found in the View's arguments.");

    $form['disable_subtotal'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable subtotal'),
      '#default_value' => $this->options['disable_subtotal'],
    ];
    $form['disable_adjustments'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable adjustments'),
      '#default_value' => $this->options['disable_adjustments'],
    ];
    $form['disable_totals'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable totals'),
      '#default_value' => $this->options['disable_totals'],
    ];
    $form['strip_trailing_zeroes'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Strip trailing zeroes after the decimal point'),
      '#default_value' => $this->options['strip_trailing_zeroes'],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE) {
    if (!$empty || !empty($this->options['empty'])) {
      foreach ($this->view->argument as $argument) {
        if (!$argument instanceof NumericArgument) {
          continue;
        }
        if (!in_array($argument->getField(), [
          'commerce_order.order_id',
          'commerce_order_item.order_id',
          'commerce_payment.order_id',
        ])) {
          continue;
        }
        if ($order = $this->orderStorage->load($argument->getValue())) {
          /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
          $order_total = $order->get('total_price')->view([
            'type' => 'commerce_configurable_order_total_summary',
            'label' => 'hidden',
            'weight' => $this->position,
            'settings' => [
              'disable_subtotal' => $this->options['disable_subtotal'],
              'disable_adjustments' => $this->options['disable_adjustments'],
              'disable_totals' => $this->options['disable_totals'],
              'strip_trailing_zeroes' => $this->options['strip_trailing_zeroes'],
            ],
          ]);
          $order_total['#prefix'] = '<div data-drupal-selector="order-total-summary">';
          $order_total['#suffix'] = '</div>';
          return $order_total;
        }
      }
    }
    return [];
  }

}
