<?php

namespace Drupal\Tests\commerce_configurable_order_total\Functional;

use Drupal\commerce_order\Adjustment;
use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_order\Entity\OrderItem;
use Drupal\commerce_price\Price;
use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\commerce_store\Entity\Store;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Drupal\views\Views;

/**
 * Functional tests for Commerce Configurable Order Total summary options.
 *
 * @group commerce_configurable_order_total
 */
class ConfigurableOrderTotalSummaryTest extends BrowserTestBase {

  /**
   * The test order.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface
   */
  protected $order;

  /**
   * The cart order to test against.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface
   */
  protected $cart;

  /**
   * The URL for the checkout form.
   *
   * @var \Drupal\Core\Url
   */
  protected $checkoutUrl;

  /**
   * Initial configurable order total settings.
   *
   * @var array
   */
  protected $initialFormatterSettings = [
    'commerce_configurable_order_total' => [
      'id' => 'commerce_configurable_order_total',
      'table' => 'views',
      'field' => 'commerce_configurable_order_total',
      'relationship' => 'none',
      'group_type' => 'group',
      'admin_label' => '',
      'plugin_id' => 'commerce_configurable_order_total',
      'empty' => TRUE,
      'disable_subtotal' => FALSE,
      'disable_adjustments' => FALSE,
      'disable_totals' => FALSE,
      'strip_trailing_zeroes' => FALSE,
    ],
  ];

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'commerce_cart',
    'commerce_checkout',
    'commerce_configurable_order_total',
    'commerce_order',
    'commerce_price',
    'commerce_product',
    'commerce_store',
    'commerce',
    'views',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->drupalLogin($this->rootUser);

    $currency_importer = $this->container->get('commerce_price.currency_importer');
    $currency_importer->import('USD');

    $store = Store::create([
      'type' => 'online',
      'name' => 'Test Store',
      'mail' => 'store@example.com',
      'default_currency' => 'USD',
      'address' => [
        'country_code' => 'US',
      ],
    ]);
    $store->save();

    $variation = ProductVariation::create([
      'type' => 'default',
      'sku' => 'TEST-SKU',
      'price' => new Price('10.00', 'USD'),
      'status' => 1,
    ]);
    $variation->save();
    $product = Product::create([
      'type' => 'default',
      'title' => 'Test Product',
      'variations' => [$variation],
      'stores' => [$store],
      'status' => 1,
    ]);
    $product->save();

    $order_item = OrderItem::create([
      'type' => 'default',
      'purchased_entity' => $variation->id(),
      'quantity' => 2,
      'unit_price' => new Price('10.00', 'USD'),
    ]);
    $order_item->addAdjustment(new Adjustment([
      'type' => 'custom',
      'label' => 'Custom fee',
      'amount' => new Price('10.00', 'USD'),
    ]));
    $order_item->save();

    $this->order = Order::create([
      'type' => 'default',
      'store_id' => $store->id(),
      'order_items' => [$order_item],
      'state' => 'draft',
      'uid' => $this->createUser(),
    ]);
    $this->order->save();

    $this->cart = $this->container->get('commerce_cart.cart_provider')->createCart('default', $store, $this->rootUser);

    $cartManager = $this->container->get('commerce_cart.cart_manager');
    $cartManager->addOrderItem($this->cart, $order_item);

    $this->checkoutUrl = Url::fromRoute('commerce_checkout.form', ['commerce_order' => $this->cart->id()]);

    $this->replaceCheckoutOrderSummaryFooter();
  }

  /**
   * Tests the default summary display (all lines shown).
   */
  public function testDefaultSummaryDisplay() {
    $this->drupalGet($this->checkoutUrl);
    $this->assertSession()->responseContains('order-total-configurable');
    $this->assertSession()->pageTextContains('Subtotal');
    $this->assertSession()->pageTextContains('Custom fee');
    $this->assertSession()->pageTextContains('Total');
    $this->assertSession()->pageTextContains('$10.00');
    $this->assertSession()->pageTextContains('$20.00');
    $this->assertSession()->pageTextContains('$30.00');
  }

  /**
   * Tests disabling the subtotal line.
   */
  public function testDisableSubtotal() {
    $this->setFormatterSetting('disable_subtotal', TRUE);
    $this->drupalGet($this->checkoutUrl);
    $this->assertSession()->pageTextNotContains('Subtotal');
  }

  /**
   * Tests disabling the adjustments line.
   */
  public function testDisableAdjustments() {
    $this->setFormatterSetting('disable_adjustments', TRUE);
    $this->drupalGet($this->checkoutUrl);
    $this->assertSession()->pageTextNotContains('Custom fee');
  }

  /**
   * Tests disabling the total line.
   */
  public function testDisableTotals() {
    $this->setFormatterSetting('disable_totals', TRUE);
    $this->drupalGet($this->checkoutUrl);
    $pageHtml = $this->getSession()->getPage()->getHtml();
    $this->assertStringNotContainsString('Total', $pageHtml);
  }

  /**
   * Tests stripping trailing zeroes from prices.
   */
  public function testStripTrailingZeroes() {
    $this->setFormatterSetting('strip_trailing_zeroes', TRUE);
    $this->drupalGet($this->checkoutUrl);
    $summary = $this->getSession()->getPage()->find('css', '[data-drupal-selector="order-total-summary"]');
    $this->assertNotNull($summary, 'Configurable order total summary element not found');
    $summaryText = $summary->getText();
    $this->assertStringContainsString('$10', $summaryText);
    $this->assertStringNotContainsString('$10.00', $summaryText);
    $this->assertStringContainsString('$20', $summaryText);
    $this->assertStringNotContainsString('$20.00', $summaryText);
    $this->assertStringContainsString('$30', $summaryText);
    $this->assertStringNotContainsString('$30.00', $summaryText);
  }

  /**
   * Replaces the commerce_order_total footer with new view.
   */
  protected function replaceCheckoutOrderSummaryFooter() {
    $view = Views::getView('commerce_checkout_order_summary');
    $this->assertNotNull($view);
    $view->initDisplay();
    $view->displayHandlers->get('default')->setOption('footer', $this->initialFormatterSettings);
    $view->save();
  }

  /**
   * Helper to set a formatter setting for the order total field.
   */
  protected function setFormatterSetting($setting, $value) {
    $view = Views::getView('commerce_checkout_order_summary');
    $view->initDisplay();
    $settings = $this->initialFormatterSettings;
    $settings['commerce_configurable_order_total'][$setting] = $value;
    $view->displayHandlers->get('default')->setOption('footer', $settings);
    $view->save();
  }

}
