<?php

namespace Drupal\Tests\commerce_configurable_order_total\Kernel\Views;

use Drupal\commerce_order\Adjustment;
use Drupal\commerce_order\Entity\Order;
use Drupal\commerce_order\Entity\OrderItem;
use Drupal\commerce_price\Price;
use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\commerce_store\Entity\Store;
use Drupal\Component\Utility\Html;
use Drupal\Core\Test\FunctionalTestSetupTrait;
use Drupal\Tests\SchemaCheckTestTrait;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;
use Drupal\views\Tests\ViewTestData;
use Drupal\views\Views;

/**
 * Tests the commerce_configurable_order_total area handler.
 *
 * @group commerce_configurable_order_total
 */
class ConfigurableOrderTotalAreaTest extends ViewsKernelTestBase {

  use FunctionalTestSetupTrait;
  use SchemaCheckTestTrait;
  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'commerce_cart',
    'commerce_checkout',
    'commerce_configurable_order_total_test',
    'commerce_configurable_order_total',
    'commerce_number_pattern',
    'commerce_order',
    'commerce_price',
    'commerce_product',
    'commerce_store',
    'commerce',
    'datetime',
    'entity_reference_revisions',
    'entity',
    'field',
    'filter',
    'inline_entity_form',
    'options',
    'path_alias',
    'path',
    'profile',
    'state_machine',
    'text',
  ];

  /**
   * Views used by this test.
   *
   * @var array
   */
  public static $testViews = ['commerce_checkout_order_summary_test'];

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\Renderer
   */
  protected $renderer;

  /**
   * The test order.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface
   */
  protected $order;

  /**
   * The cart order to test against.
   *
   * @var \Drupal\commerce_order\Entity\OrderInterface
   */
  protected $cart;

  /**
   * Initial area settings.
   *
   * @var array
   */
  protected $initialAreaSettings = [
    'commerce_configurable_order_total' => [
      'id' => 'commerce_configurable_order_total',
      'table' => 'views',
      'field' => 'commerce_configurable_order_total',
      'relationship' => 'none',
      'group_type' => 'group',
      'admin_label' => '',
      'plugin_id' => 'commerce_configurable_order_total',
      'empty' => TRUE,
      'disable_subtotal' => FALSE,
      'disable_adjustments' => FALSE,
      'disable_totals' => FALSE,
      'strip_trailing_zeroes' => FALSE,
    ],
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = TRUE): void {
    parent::setUp($import_test_views);

    $this->renderer = $this->container->get('renderer');

    $this->installEntitySchema('path_alias');
    $this->installEntitySchema('commerce_currency');
    $this->installEntitySchema('commerce_store');
    $this->installConfig(['commerce_store']);
    $this->installEntitySchema('user');
    $this->installEntitySchema('profile');
    $this->installEntitySchema('commerce_order');
    $this->installEntitySchema('commerce_order_item');
    $this->installEntitySchema('commerce_product');
    $this->installEntitySchema('commerce_product_variation');
    $this->installConfig(['commerce_product', 'commerce_order', 'system']);
    $this->installSchema('commerce_number_pattern', ['commerce_number_pattern_sequence']);

    $currency_importer = $this->container->get('commerce_price.currency_importer');
    $currency_importer->import('USD');

    $store = Store::create([
      'type' => 'online',
      'name' => 'Test Store',
      'mail' => 'store@example.com',
      'default_currency' => 'USD',
      'address' => [
        'country_code' => 'US',
      ],
    ]);
    $store->save();

    $variation = ProductVariation::create([
      'type' => 'default',
      'sku' => 'TEST-SKU',
      'price' => new Price('10.00', 'USD'),
      'status' => 1,
    ]);
    $variation->save();
    $product = Product::create([
      'type' => 'default',
      'title' => 'Test Product',
      'variations' => [$variation],
      'stores' => [$store],
      'status' => 1,
    ]);
    $product->save();

    $order_item = OrderItem::create([
      'type' => 'default',
      'purchased_entity' => $variation->id(),
      'quantity' => 2,
      'unit_price' => new Price('10.00', 'USD'),
    ]);
    $order_item->addAdjustment(new Adjustment([
      'type' => 'custom',
      'label' => 'Custom fee',
      'amount' => new Price('10.00', 'USD'),
    ]));
    $order_item->save();

    $this->order = Order::create([
      'type' => 'default',
      'store_id' => $store->id(),
      'order_items' => [$order_item],
      'state' => 'draft',
      'uid' => $this->createUser(),
    ]);
    $this->order->save();

    $this->cart = $this->container->get('commerce_cart.cart_provider')->createCart('default', $store, $this->rootUser);

    $cartManager = $this->container->get('commerce_cart.cart_manager');
    $cartManager->addOrderItem($this->cart, $order_item);

    ViewTestData::createTestViews(self::class, ['commerce_configurable_order_total_test']);
  }

  /**
   * Tests the order summary view.
   */
  public function testOrderSummaryView() {
    $view = $this->getOrderSummaryView();
    $this->assertNotNull($view, 'The view commerce_checkout_order_summary_test was not found.');
    $this->assertConfigSchemaByName('views.view.commerce_configurable_order_total_test');
  }

  /**
   * Tests the commerce_configurable_order_total area handler.
   */
  public function testAreaHandler() {
    $output = $this->getOrderSummaryOutput();
    $this->assertStringContainsString('order-total-configurable', $output);
    $this->assertStringContainsString('Subtotal', $output);
    $this->assertStringContainsString('Custom fee', $output);
    $this->assertStringContainsString('Total', $output);
    $this->assertStringContainsString('$10.00', $output);
    $this->assertStringContainsString('$20.00', $output);
    $this->assertStringContainsString('$30.00', $output);
  }

  /**
   * Tests disabling the subtotal line.
   */
  public function testDisableSubtotal() {
    $this->setAreaSetting('disable_subtotal', TRUE);
    $output = $this->getOrderSummaryOutput();
    $this->assertStringNotContainsString('Subtotal', $output);
  }

  /**
   * Tests disabling the adjustments line.
   */
  public function testDisableAdjustments() {
    $this->setAreaSetting('disable_adjustments', TRUE);
    $output = $this->getOrderSummaryOutput();
    $this->assertStringNotContainsString('Custom fee', $output);
  }

  /**
   * Tests disabling the total line.
   */
  public function testDisableTotals() {
    $this->setAreaSetting('disable_totals', TRUE);
    $output = $this->getOrderSummaryOutput();
    $this->assertStringNotContainsString('Total', $output);
  }

  /**
   * Tests stripping trailing zeroes from prices.
   */
  public function testStripTrailingZeroes() {
    $this->setAreaSetting('strip_trailing_zeroes', TRUE);
    $output = $this->getOrderSummaryOutput();

    $dom = Html::load($output);
    $xpath = new \DOMXPath($dom);
    $nodes = $xpath->query('//div[@data-drupal-selector="order-total-summary"]');
    if ($nodes->length > 0) {
      $summaryContent = $dom->saveHTML($nodes->item(0));
      $this->assertStringContainsString('$10', $summaryContent);
      $this->assertStringNotContainsString('$10.00', $summaryContent);
      $this->assertStringContainsString('$20', $summaryContent);
      $this->assertStringNotContainsString('$20.00', $summaryContent);
      $this->assertStringContainsString('$30', $summaryContent);
      $this->assertStringNotContainsString('$30.00', $summaryContent);
    }
    else {
      $this->fail('Order total summary element not found');
    }

  }

  /**
   * Helper to set a area setting.
   */
  protected function setAreaSetting($setting, $value) {
    $view = $this->getOrderSummaryView();
    $settings = $this->initialAreaSettings;
    $settings['commerce_configurable_order_total'][$setting] = $value;
    $view->displayHandlers->get('default')->setOption('footer', $settings);
    $view->save();
  }

  /**
   * Get the order summary output.
   */
  protected function getOrderSummaryOutput() {
    $view = $this->getOrderSummaryView();
    $output = $view->preview();
    $output = (string) $this->renderer->renderRoot($output);
    $this->assertNotEmpty($output);
    return $output;
  }

  /**
   * Get the order summary view.
   */
  protected function getOrderSummaryView() {
    $view = Views::getView('commerce_checkout_order_summary_test');
    $view->initHandlers();
    $view->setArguments([$this->cart->id()]);
    $view->initDisplay();
    return $view;
  }

}
