<?php

namespace Drupal\commerce_currency_mismatch_prevention\Dto;

/**
 * Data transfer object for currency validation results.
 */
class CurrencyValidationResult {

  /**
   * Constructs a new CurrencyValidationResult.
   *
   * @param bool $valid
   *   Whether the currency is valid (no conflict).
   * @param string|null $newCurrency
   *   The currency code of the new item being added.
   * @param string|null $existingCurrency
   *   The currency code of existing items in cart (if conflict exists).
   */
  public function __construct(
    protected bool $valid,
    protected ?string $newCurrency = NULL,
    protected ?string $existingCurrency = NULL,
  ) {
  }

  /**
   * Checks if validation passed (no currency conflict).
   *
   * @return bool
   *   TRUE if valid, FALSE if conflict detected.
   */
  public function isValid(): bool {
    return $this->valid;
  }

  /**
   * Checks if there is a currency conflict.
   *
   * @return bool
   *   TRUE if conflict exists, FALSE otherwise.
   */
  public function hasConflict(): bool {
    return !$this->valid;
  }

  /**
   * Gets the new currency code.
   *
   * @return string|null
   *   The currency code, or NULL if not available.
   */
  public function getNewCurrency(): ?string {
    return $this->newCurrency;
  }

  /**
   * Gets the existing currency code.
   *
   * @return string|null
   *   The currency code, or NULL if no conflict or no existing items.
   */
  public function getExistingCurrency(): ?string {
    return $this->existingCurrency;
  }

}
