<?php

namespace Drupal\commerce_currency_mismatch_prevention\Service;

use Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for handling currency conflict messages.
 */
class CurrencyMessageService {

  use StringTranslationTrait;

  /**
   * Constructs a new CurrencyMessageService.
   *
   * @param \Drupal\commerce_currency_mismatch_prevention\Service\SettingsService $settings
   *   The settings service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(
    protected SettingsService $settings,
    protected MessengerInterface $messenger,
  ) {
  }

  /**
   * Shows a message about cart being emptied to accommodate new currency.
   *
   * @param \Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult|null $validation
   *   The validation result containing currency information (optional).
   */
  public function showCartEmptiedMessage(?CurrencyValidationResult $validation = NULL): void {
    $oldCurrency = $validation?->getExistingCurrency();
    $newCurrency = $validation?->getNewCurrency();

    if ($oldCurrency && $newCurrency) {
      $message = $this->t('Your previous @old_currency items have been removed from the cart to add the @new_currency product.', [
        '@old_currency' => $oldCurrency,
        '@new_currency' => $newCurrency,
      ]);
    }
    else {
      $message = $this->t('Your cart has been emptied to accommodate products in a different currency.');
    }

    $this->showMessage($message);
  }

  /**
   * Shows a message about product not being added due to currency conflict.
   *
   * @param \Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult|null $validation
   *   The validation result containing currency information (optional).
   */
  public function showProductRejectedMessage(?CurrencyValidationResult $validation = NULL): void {
    $existingCurrency = $validation?->getExistingCurrency();
    $newCurrency = $validation?->getNewCurrency();

    if ($existingCurrency && $newCurrency) {
      $message = $this->t('The @new_currency product was not added because your cart contains @existing_currency items. Please complete your current order first.', [
        '@existing_currency' => $existingCurrency,
        '@new_currency' => $newCurrency,
      ]);
    }
    else {
      $message = $this->t('This product cannot be added because your cart contains items in a different currency. Please complete your current order first.');
    }

    $this->showMessage($message);
  }

  /**
   * Shows a message based on behavior type.
   *
   * @param \Drupal\Core\StringTranslation\TranslatableMarkup|string $message
   *   The message to display.
   * @param string $messageType
   *   The message type.
   */
  public function showMessage($message, ?string $messageType = NULL): void {
    if (!$this->settings->isUserNotificationEnabled()) {
      return;
    }

    $messageType ??= $this->settings->getNotificationMessageType();

    switch ($messageType) {
      case 'error':
        $this->messenger->addError($message);
        break;

      case 'status':
        $this->messenger->addStatus($message);
        break;

      default:
        $this->messenger->addWarning($message);
        break;
    }
  }

}
