<?php

namespace Drupal\commerce_currency_mismatch_prevention;

use Drupal\commerce_cart\CartManagerInterface;
use Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult;
use Drupal\commerce_currency_mismatch_prevention\Service\SettingsService;
use Drupal\commerce_currency_mismatch_prevention\Service\CurrencyValidationService;
use Drupal\commerce_currency_mismatch_prevention\Service\CurrencyMessageService;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\commerce_price\Exception\CurrencyMismatchException;
use Drupal\commerce\PurchasableEntityInterface;
use Psr\Log\LoggerInterface;

/**
 * Decorates the cart manager to handle currency mismatches.
 */
class CartManagerDecorator implements CartManagerInterface {

  /**
   * Constructs a new CartManagerDecorator.
   *
   * @param \Drupal\commerce_cart\CartManagerInterface $cartManager
   *   The decorated cart manager.
   * @param \Drupal\commerce_currency_mismatch_prevention\Service\CurrencyValidationService $validationService
   *   The currency validation service.
   * @param \Drupal\commerce_currency_mismatch_prevention\Service\CurrencyMessageService $messageService
   *   The currency message service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\commerce_currency_mismatch_prevention\Service\SettingsService $settings
   *   The settings service.
   */
  public function __construct(
    protected CartManagerInterface $cartManager,
    protected CurrencyValidationService $validationService,
    protected CurrencyMessageService $messageService,
    protected LoggerInterface $logger,
    protected SettingsService $settings,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function emptyCart(OrderInterface $cart, $save_cart = TRUE) {
    return $this->cartManager->emptyCart($cart, $save_cart);
  }

  /**
   * {@inheritdoc}
   */
  public function addEntity(OrderInterface $cart, PurchasableEntityInterface $entity, $quantity = '1', $combine = TRUE, $save_cart = TRUE) {
    // Pre-validate currency BEFORE creating order item.
    $validation = $this->validationService->validateEntity($cart, $entity);

    if (!$validation->isValid()) {
      return $this->handleEntityValidationFailure($cart, $entity, $quantity, $combine, $save_cart, $validation);
    }

    try {
      return $this->cartManager->addEntity($cart, $entity, $quantity, $combine, $save_cart);
    }
    catch (CurrencyMismatchException $e) {
      return $this->handleEntityException($cart, $entity, $quantity, $combine, $save_cart, $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createOrderItem(PurchasableEntityInterface $entity, $quantity = '1') {
    return $this->cartManager->createOrderItem($entity, $quantity);
  }

  /**
   * {@inheritdoc}
   */
  public function addOrderItem(OrderInterface $cart, OrderItemInterface $order_item, $combine = TRUE, $save_cart = TRUE) {
    // Pre-validate currency BEFORE CartManager saves the order item.
    $validation = $this->validationService->validateOrderItem($cart, $order_item);

    if (!$validation->isValid()) {
      return $this->handleOrderItemValidationFailure($cart, $order_item, $combine, $save_cart, $validation);
    }

    try {
      return $this->cartManager->addOrderItem($cart, $order_item, $combine, $save_cart);
    }
    catch (CurrencyMismatchException $e) {
      return $this->handleOrderItemException($cart, $order_item, $combine, $save_cart, $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function updateOrderItem(OrderInterface $cart, OrderItemInterface $order_item, $save_cart = TRUE) {
    return $this->cartManager->updateOrderItem($cart, $order_item, $save_cart);
  }

  /**
   * {@inheritdoc}
   */
  public function removeOrderItem(OrderInterface $cart, OrderItemInterface $order_item, $save_cart = TRUE) {
    return $this->cartManager->removeOrderItem($cart, $order_item, $save_cart);
  }

  /**
   * Handles validation failure when adding an entity.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order.
   * @param \Drupal\commerce\PurchasableEntityInterface $entity
   *   The purchasable entity.
   * @param string $quantity
   *   The quantity.
   * @param bool $combine
   *   Whether to combine order items.
   * @param bool $save_cart
   *   Whether to save the cart.
   * @param \Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult $validation
   *   The validation result.
   *
   * @return \Drupal\commerce_order\Entity\OrderItemInterface|null
   *   The order item, or NULL if rejected.
   */
  protected function handleEntityValidationFailure(OrderInterface $cart, PurchasableEntityInterface $entity, $quantity, $combine, $save_cart, CurrencyValidationResult $validation) {
    $behavior = $this->settings->getCartBehavior();

    if ($behavior === 'remove_existing') {
      // Empty cart and retry.
      $this->messageService->showCartEmptiedMessage($validation);
      $this->cartManager->emptyCart($cart, FALSE);
      return $this->cartManager->addEntity($cart, $entity, $quantity, $combine, $save_cart);
    }

    // keep_current - reject addition.
    $this->messageService->showProductRejectedMessage($validation);
    return NULL;
  }

  /**
   * Handles currency mismatch exception when adding an entity.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order.
   * @param \Drupal\commerce\PurchasableEntityInterface $entity
   *   The purchasable entity.
   * @param string $quantity
   *   The quantity.
   * @param bool $combine
   *   Whether to combine order items.
   * @param bool $save_cart
   *   Whether to save the cart.
   * @param \Drupal\commerce_price\Exception\CurrencyMismatchException $e
   *   The exception.
   *
   * @return \Drupal\commerce_order\Entity\OrderItemInterface|null
   *   The order item, or NULL if rejected.
   */
  protected function handleEntityException(OrderInterface $cart, PurchasableEntityInterface $entity, $quantity, $combine, $save_cart, CurrencyMismatchException $e) {
    $behavior = $this->settings->getCartBehavior();

    if ($behavior === 'disabled') {
      throw $e;
    }

    if ($behavior === 'remove_existing') {
      $this->messageService->showCartEmptiedMessage();
      $this->cartManager->emptyCart($cart, FALSE);
      return $this->cartManager->addEntity($cart, $entity, $quantity, $combine, $save_cart);
    }

    // keep_current - show error and clean up.
    $this->messageService->showProductRejectedMessage();
    $this->cleanupPartiallyCreatedOrderItem($cart, $entity);
    return NULL;
  }

  /**
   * Handles validation failure when adding an order item.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order.
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $order_item
   *   The order item.
   * @param bool $combine
   *   Whether to combine order items.
   * @param bool $save_cart
   *   Whether to save the cart.
   * @param \Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult $validation
   *   The validation result.
   *
   * @return \Drupal\commerce_order\Entity\OrderItemInterface
   *   The order item.
   */
  protected function handleOrderItemValidationFailure(OrderInterface $cart, OrderItemInterface $order_item, $combine, $save_cart, CurrencyValidationResult $validation) {
    $behavior = $this->settings->getCartBehavior();

    if ($behavior === 'remove_existing') {
      // Empty cart and retry.
      $this->messageService->showCartEmptiedMessage($validation);
      $this->cartManager->emptyCart($cart, FALSE);
      return $this->cartManager->addOrderItem($cart, $order_item, $combine, $save_cart);
    }

    // keep_current - reject addition.
    $this->messageService->showProductRejectedMessage($validation);
    return $order_item;
  }

  /**
   * Handles currency mismatch exception when adding an order item.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order.
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $order_item
   *   The order item.
   * @param bool $combine
   *   Whether to combine order items.
   * @param bool $save_cart
   *   Whether to save the cart.
   * @param \Drupal\commerce_price\Exception\CurrencyMismatchException $e
   *   The exception.
   *
   * @return \Drupal\commerce_order\Entity\OrderItemInterface
   *   The order item.
   */
  protected function handleOrderItemException(OrderInterface $cart, OrderItemInterface $order_item, $combine, $save_cart, CurrencyMismatchException $e) {
    $behavior = $this->settings->getCartBehavior();

    if ($behavior === 'disabled') {
      throw $e;
    }

    if ($behavior === 'remove_existing') {
      $this->messageService->showCartEmptiedMessage();
      $this->cartManager->emptyCart($cart, FALSE);
      return $this->cartManager->addOrderItem($cart, $order_item, $combine, $save_cart);
    }

    // keep_current - show error and clean up.
    $this->messageService->showProductRejectedMessage();
    $this->cleanupSavedOrderItem($cart, $order_item);
    return $order_item;
  }

  /**
   * Cleans up partially created order items after a failed add.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order.
   * @param \Drupal\commerce\PurchasableEntityInterface $entity
   *   The purchasable entity.
   */
  protected function cleanupPartiallyCreatedOrderItem(OrderInterface $cart, PurchasableEntityInterface $entity): void {
    $orderItems = $cart->getItems();
    $entityId = $entity->id();
    $entityType = $entity->getEntityTypeId();

    foreach ($orderItems as $orderItem) {
      $purchasedEntity = $orderItem->getPurchasedEntity();
      if ($purchasedEntity &&
          $purchasedEntity->id() == $entityId &&
          $purchasedEntity->getEntityTypeId() == $entityType &&
          $orderItem->isNew()) {
        try {
          $cart->removeItem($orderItem);
          $orderItem->delete();
        }
        catch (\Exception $deleteException) {
          $this->logger->error('Failed to clean up order item: @message', [
            '@message' => $deleteException->getMessage(),
          ]);
        }
      }
    }
  }

  /**
   * Cleans up a saved order item after rejection.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order.
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $order_item
   *   The order item.
   */
  protected function cleanupSavedOrderItem(OrderInterface $cart, OrderItemInterface $order_item): void {
    try {
      if (!$order_item->isNew()) {
        if ($cart->hasItem($order_item)) {
          $cart->removeItem($order_item);
        }
        $order_item->delete();
      }
    }
    catch (\Exception $deleteException) {
      $this->logger->error('Failed to delete rejected order item: @message', [
        '@message' => $deleteException->getMessage(),
      ]);
    }
  }

}
