<?php

namespace Drupal\commerce_currency_mismatch_prevention\Form;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Commerce currency mismatch prevention settings form.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const SETTINGS = 'commerce_currency_mismatch_prevention.settings';

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   */
  public function __construct(ModuleHandlerInterface $module_handler) {
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'commerce_currency_mismatch_prevention_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::SETTINGS,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::SETTINGS);

    $form['cart_management'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('Cart currency management'),
      '#description' => $this->t('Configure how to handle products with different currencies in the cart.'),
    ];

    $form['cart_management']['remove_different_currency'] = [
      '#type' => 'radios',
      '#title' => $this->t('Currency conflict behavior'),
      '#description' => $this->t('What to do when adding a product with different currency to the cart.'),
      '#default_value' => $config->get('cart_management.remove_different_currency') ?: 'remove_existing',
      '#options' => [
        'keep_current' => $this->t('Remove new product, keep current cart items'),
        'remove_existing' => $this->t('Remove existing cart items, add new product'),
        'disabled' => $this->t('Disabled - allow currency mismatch (may cause errors)'),
      ],
    ];

    $form['notifications'] = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->t('Notification settings'),
      '#description' => $this->t('Configure user notifications for currency conflicts.'),
      '#states' => [
        'invisible' => [
          [':input[name="remove_different_currency"]' => ['value' => 'disabled']],
        ],
      ],
    ];

    $form['notifications']['show_user_message'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show user notification'),
      '#description' => $this->t('Display a message to users when currency conflicts occur.'),
      '#default_value' => $config->get('notifications.show_user_message') ?? FALSE,
    ];

    $form['notifications']['message_type'] = [
      '#type' => 'radios',
      '#title' => $this->t('Message type'),
      '#default_value' => $config->get('notifications.message_type') ?: 'status',
      '#options' => [
        'warning' => $this->t('Warning'),
        'status' => $this->t('Status'),
        'error' => $this->t('Error'),
      ],
      '#states' => [
        'visible' => [
          ':input[name="show_user_message"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config(static::SETTINGS)
      ->set('cart_management.remove_different_currency', $form_state->getValue('remove_different_currency'))
      ->set('notifications.show_user_message', $form_state->getValue('show_user_message'))
      ->set('notifications.message_type', $form_state->getValue('message_type'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
