<?php

namespace Drupal\commerce_currency_mismatch_prevention\Service;

use Drupal\commerce\PurchasableEntityInterface;
use Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyConflictResult;
use Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Entity\OrderItemInterface;

/**
 * Service for validating currency compatibility in shopping cart.
 */
class CurrencyValidationService {

  /**
   * Constructs a new CurrencyValidationService.
   *
   * @param \Drupal\commerce_currency_mismatch_prevention\Service\SettingsService $settings
   *   The settings service.
   */
  public function __construct(
    protected SettingsService $settings,
  ) {
  }

  /**
   * Detects currency conflicts in cart for a new currency.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order.
   * @param string $newCurrency
   *   The currency code of the item being added.
   *
   * @return \Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyConflictResult
   *   The conflict detection result.
   */
  public function detectConflict(OrderInterface $cart, string $newCurrency): CurrencyConflictResult {
    $cartItems = $cart->getItems();
    if (empty($cartItems)) {
      return new CurrencyConflictResult(FALSE);
    }

    foreach ($cartItems as $cartItem) {
      $unitPrice = $cartItem->getUnitPrice();
      if (!$unitPrice) {
        continue;
      }

      $existingCurrency = $unitPrice->getCurrencyCode();
      if ($existingCurrency === $newCurrency) {
        continue;
      }

      return new CurrencyConflictResult(
        hasConflict: TRUE,
        existingCurrency: $existingCurrency,
      );
    }

    return new CurrencyConflictResult(FALSE);
  }

  /**
   * Gets the currency code from a purchasable entity.
   *
   * @param \Drupal\commerce\PurchasableEntityInterface $entity
   *   The purchasable entity.
   *
   * @return string|null
   *   The currency code, or NULL if no price available.
   */
  public function getCurrencyFromEntity(PurchasableEntityInterface $entity): ?string {
    $price = $entity->getPrice();
    return $price ? $price->getCurrencyCode() : NULL;
  }

  /**
   * Gets the currency code from an order item.
   *
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $orderItem
   *   The order item.
   *
   * @return string|null
   *   The currency code, or NULL if no price available.
   */
  public function getCurrencyFromOrderItem(OrderItemInterface $orderItem): ?string {
    $price = $orderItem->getUnitPrice();
    return $price ? $price->getCurrencyCode() : NULL;
  }

  /**
   * Validates currency compatibility for a purchasable entity.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order.
   * @param \Drupal\commerce\PurchasableEntityInterface $entity
   *   The purchasable entity to validate.
   *
   * @return \Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult
   *   The validation result.
   */
  public function validateEntity(OrderInterface $cart, PurchasableEntityInterface $entity): CurrencyValidationResult {
    if (!$this->settings->isEnabled()) {
      return new CurrencyValidationResult(TRUE);
    }

    $newCurrency = $this->getCurrencyFromEntity($entity);
    if (!$newCurrency) {
      // No price, cannot validate.
      return new CurrencyValidationResult(TRUE);
    }

    $conflict = $this->detectConflict($cart, $newCurrency);

    return new CurrencyValidationResult(
      valid: !$conflict->hasConflict(),
      newCurrency: $newCurrency,
      existingCurrency: $conflict->getExistingCurrency(),
    );
  }

  /**
   * Validates currency compatibility for an order item.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $cart
   *   The cart order.
   * @param \Drupal\commerce_order\Entity\OrderItemInterface $orderItem
   *   The order item to validate.
   *
   * @return \Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult
   *   The validation result.
   */
  public function validateOrderItem(OrderInterface $cart, OrderItemInterface $orderItem): CurrencyValidationResult {
    if (!$this->settings->isEnabled()) {
      return new CurrencyValidationResult(TRUE);
    }

    $newCurrency = $this->getCurrencyFromOrderItem($orderItem);
    if (!$newCurrency) {
      // No price, cannot validate.
      return new CurrencyValidationResult(TRUE);
    }

    $conflict = $this->detectConflict($cart, $newCurrency);

    return new CurrencyValidationResult(
      valid: !$conflict->hasConflict(),
      newCurrency: $newCurrency,
      existingCurrency: $conflict->getExistingCurrency(),
    );
  }

}
