<?php

namespace Drupal\commerce_currency_mismatch_prevention\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for managing Commerce currency mismatch prevention settings.
 */
class SettingsService {

  /**
   * Config settings name.
   */
  const SETTINGS = 'commerce_currency_mismatch_prevention.settings';

  /**
   * Constructs a new SettingsService.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * Gets the configured behavior for currency conflicts.
   *
   * @return string
   *   The behavior: 'disabled', 'keep_current', or 'remove_existing'.
   */
  public function getCartBehavior(): string {
    $config = $this->configFactory->get(self::SETTINGS);
    return $config->get('cart_management.remove_different_currency') ?? 'disabled';
  }

  /**
   * Checks if currency mismatch prevention is enabled.
   *
   * @return bool
   *   TRUE if enabled, FALSE otherwise.
   */
  public function isEnabled(): bool {
    return $this->getCartBehavior() !== 'disabled';
  }

  /**
   * Checks if user notifications are enabled.
   *
   * @return bool
   *   TRUE if enabled, FALSE otherwise.
   */
  public function isUserNotificationEnabled(): bool {
    $config = $this->configFactory->get(self::SETTINGS);
    return (bool) $config->get('notifications.show_user_message') ?? FALSE;
  }

  /**
   * Gets the notification message type.
   *
   * @return string
   *   The message type: 'error', 'warning', or 'status'.
   */
  public function getNotificationMessageType(): string {
    $config = $this->configFactory->get(self::SETTINGS);
    return $config->get('notifications.message_type') ?? 'status';
  }

}
