<?php

namespace Drupal\Tests\commerce_currency_mismatch_prevention\Kernel;

use Drupal\commerce_currency_mismatch_prevention\CartManagerDecorator;
use Drupal\commerce_order\Entity\OrderItem;
use Drupal\commerce_price\Exception\CurrencyMismatchException;
use Drupal\commerce_price\Price;
use Drupal\commerce_product\Entity\Product;
use Drupal\commerce_product\Entity\ProductVariation;
use Drupal\commerce_store\StoreCreationTrait;
use Drupal\KernelTests\KernelTestBase;
use Drupal\user\Entity\User;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the cart manager decorator integration.
 */
#[CoversClass(CartManagerDecorator::class)]
#[Group('commerce_currency_mismatch_prevention')]
class CartManagerDecoratorKernelTest extends KernelTestBase {

  use StoreCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'field',
    'options',
    'user',
    'path_alias',
    'text',
    'address',
    'datetime',
    'entity',
    'inline_entity_form',
    'views',
    'commerce',
    'commerce_price',
    'commerce_store',
    'path',
    'entity_reference_revisions',
    'profile',
    'state_machine',
    'commerce_number_pattern',
    'commerce_product',
    'commerce_order',
    'commerce_cart',
    'commerce_currency_mismatch_prevention',
  ];

  /**
   * The cart manager.
   *
   * @var \Drupal\commerce_cart\CartManagerInterface
   */
  protected $cartManager;

  /**
   * The cart provider.
   *
   * @var \Drupal\commerce_cart\CartProviderInterface
   */
  protected $cartProvider;

  /**
   * A sample user.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $user;

  /**
   * A sample store.
   *
   * @var \Drupal\commerce_store\Entity\StoreInterface
   */
  protected $store;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('commerce_currency');
    $this->installEntitySchema('commerce_store');
    $this->installEntitySchema('commerce_product_variation');
    $this->installEntitySchema('commerce_product');
    $this->installEntitySchema('commerce_order');
    $this->installEntitySchema('commerce_order_item');
    $this->installEntitySchema('path_alias');
    $this->installConfig('commerce_order');
    $this->installConfig('commerce_cart');
    $this->installConfig('commerce_product');
    $this->installConfig('commerce_currency_mismatch_prevention');

    $currency_importer = \Drupal::service('commerce_price.currency_importer');
    $currency_importer->import('USD');
    $currency_importer->import('EUR');

    $user = User::create([
      'name' => 'test_user',
      'mail' => 'test@example.com',
    ]);
    $user->save();
    $this->user = $user;

    $this->store = $this->createStore('Default store', 'admin@example.com', 'online', TRUE, 'US', 'USD');

    $this->cartManager = \Drupal::service('commerce_cart.cart_manager');
    $this->cartProvider = \Drupal::service('commerce_cart.cart_provider');
  }

  /**
   * Creates a product variation.
   *
   * @param string $price
   *   The price amount.
   * @param string $currency
   *   The currency code.
   *
   * @return \Drupal\commerce_product\Entity\ProductVariationInterface
   *   The product variation.
   */
  protected function createProductVariation($price, $currency) {
    $product = Product::create([
      'type' => 'default',
      'title' => $this->randomString(),
      'stores' => [$this->store],
    ]);
    $product->save();

    $variation = ProductVariation::create([
      'type' => 'default',
      'product_id' => $product->id(),
      'sku' => $this->randomString(),
      'price' => new Price($price, $currency),
    ]);
    $variation->save();

    return $variation;
  }

  /**
   * Tests module disabled behavior - should throw CurrencyMismatchException.
   */
  public function testModuleDisabledThrowsException(): void {
    $config = \Drupal::configFactory()->getEditable('commerce_currency_mismatch_prevention.settings');
    $config->set('cart_management.remove_different_currency', 'disabled')->save();

    $cart = $this->cartProvider->createCart('default', $this->store, $this->user);

    $usd_variation = $this->createProductVariation('10.00', 'USD');
    $this->cartManager->addEntity($cart, $usd_variation);

    $this->assertCount(1, $cart->getItems());
    $this->assertEquals('USD', $cart->getItems()[0]->getUnitPrice()->getCurrencyCode());

    $eur_variation = $this->createProductVariation('15.00', 'EUR');

    $this->expectException(CurrencyMismatchException::class);
    $this->cartManager->addEntity($cart, $eur_variation);
  }

  /**
   * Tests remove_existing behavior - should empty cart and add new product.
   */
  public function testRemoveExistingEmptiesCartAndAddsProduct(): void {
    $config = \Drupal::configFactory()->getEditable('commerce_currency_mismatch_prevention.settings');
    $config->set('cart_management.remove_different_currency', 'remove_existing')->save();

    $cart = $this->cartProvider->createCart('default', $this->store, $this->user);

    $usd_variation = $this->createProductVariation('10.00', 'USD');
    $this->cartManager->addEntity($cart, $usd_variation);

    $this->assertCount(1, $cart->getItems());
    $this->assertEquals('USD', $cart->getItems()[0]->getUnitPrice()->getCurrencyCode());

    // Add EUR product - should remove USD and add EUR.
    $eur_variation = $this->createProductVariation('15.000000', 'EUR');
    $order_item = $this->cartManager->addEntity($cart, $eur_variation);

    $this->assertNotNull($order_item);
    $this->assertCount(1, $cart->getItems());
    $this->assertEquals('EUR', $cart->getItems()[0]->getUnitPrice()->getCurrencyCode());
    $this->assertEquals('15.000000', $cart->getItems()[0]->getUnitPrice()->getNumber());
  }

  /**
   * Tests keep_current behavior - should reject new product.
   */
  public function testKeepCurrentRejectsNewProduct(): void {
    $config = \Drupal::configFactory()->getEditable('commerce_currency_mismatch_prevention.settings');
    $config->set('cart_management.remove_different_currency', 'keep_current')->save();

    $cart = $this->cartProvider->createCart('default', $this->store, $this->user);

    $usd_variation = $this->createProductVariation('10.00', 'USD');
    $this->cartManager->addEntity($cart, $usd_variation);

    $this->assertCount(1, $cart->getItems());
    $this->assertEquals('USD', $cart->getItems()[0]->getUnitPrice()->getCurrencyCode());

    // Try to add EUR product - should be rejected.
    $eur_variation = $this->createProductVariation('15.000000', 'EUR');
    $order_item = $this->cartManager->addEntity($cart, $eur_variation);

    $this->assertNull($order_item);
    $this->assertCount(1, $cart->getItems());
    $this->assertEquals('USD', $cart->getItems()[0]->getUnitPrice()->getCurrencyCode());
    $this->assertEquals('10.000000', $cart->getItems()[0]->getUnitPrice()->getNumber());
  }

  /**
   * Tests same currency products are allowed regardless of behavior.
   */
  public function testSameCurrencyProductsAlwaysAllowed(): void {
    $config = \Drupal::configFactory()->getEditable('commerce_currency_mismatch_prevention.settings');
    $config->set('cart_management.remove_different_currency', 'remove_existing')->save();

    $cart = $this->cartProvider->createCart('default', $this->store, $this->user);

    $usd_variation1 = $this->createProductVariation('10.00', 'USD');
    $this->cartManager->addEntity($cart, $usd_variation1);

    $usd_variation2 = $this->createProductVariation('20.00', 'USD');
    $order_item = $this->cartManager->addEntity($cart, $usd_variation2);

    $this->assertNotNull($order_item);
    $this->assertCount(2, $cart->getItems());

    foreach ($cart->getItems() as $item) {
      $this->assertEquals('USD', $item->getUnitPrice()->getCurrencyCode());
    }
  }

  /**
   * Tests addOrderItem with currency conflicts.
   */
  public function testAddOrderItemWithCurrencyConflicts(): void {
    $config = \Drupal::configFactory()->getEditable('commerce_currency_mismatch_prevention.settings');
    $config->set('cart_management.remove_different_currency', 'keep_current')->save();

    $cart = $this->cartProvider->createCart('default', $this->store, $this->user);

    $usd_variation = $this->createProductVariation('10.00', 'USD');
    $this->cartManager->addEntity($cart, $usd_variation);

    $eur_variation = $this->createProductVariation('15.000000', 'EUR');
    $eur_order_item = OrderItem::create([
      'type' => 'default',
      'purchased_entity' => $eur_variation,
      'quantity' => '1',
      'unit_price' => new Price('15.000000', 'EUR'),
    ]);
    $eur_order_item->save();

    // Try to add EUR order item - should be rejected.
    $result = $this->cartManager->addOrderItem($cart, $eur_order_item);

    $this->assertEquals($eur_order_item, $result);
    $this->assertCount(1, $cart->getItems());
    $this->assertEquals('USD', $cart->getItems()[0]->getUnitPrice()->getCurrencyCode());
  }

  /**
   * Tests empty cart allows any currency.
   */
  public function testEmptyCartAllowsAnyCurrency(): void {
    $config = \Drupal::configFactory()->getEditable('commerce_currency_mismatch_prevention.settings');
    $config->set('cart_management.remove_different_currency', 'keep_current')->save();

    $cart = $this->cartProvider->createCart('default', $this->store, $this->user);

    // Add EUR product to empty cart - should be allowed.
    $eur_variation = $this->createProductVariation('15.000000', 'EUR');
    $order_item = $this->cartManager->addEntity($cart, $eur_variation);

    $this->assertNotNull($order_item);
    $this->assertCount(1, $cart->getItems());
    $this->assertEquals('EUR', $cart->getItems()[0]->getUnitPrice()->getCurrencyCode());
  }

  /**
   * Tests configuration changes take effect immediately.
   */
  public function testConfigurationChanges(): void {
    $cart = $this->cartProvider->createCart('default', $this->store, $this->user);

    $usd_variation = $this->createProductVariation('10.00', 'USD');
    $this->cartManager->addEntity($cart, $usd_variation);

    $eur_variation = $this->createProductVariation('15.000000', 'EUR');

    // Test with disabled - should throw exception.
    $config = \Drupal::configFactory()->getEditable('commerce_currency_mismatch_prevention.settings');
    $config->set('cart_management.remove_different_currency', 'disabled')->save();

    $this->expectException(CurrencyMismatchException::class);
    $this->cartManager->addEntity($cart, $eur_variation);
  }

}
