<?php

namespace Drupal\Tests\commerce_currency_mismatch_prevention\Unit;

use Drupal\commerce\PurchasableEntityInterface;
use Drupal\commerce_cart\CartManagerInterface;
use Drupal\commerce_currency_mismatch_prevention\CartManagerDecorator;
use Drupal\commerce_currency_mismatch_prevention\Dto\CurrencyValidationResult;
use Drupal\commerce_currency_mismatch_prevention\Service\CurrencyMessageService;
use Drupal\commerce_currency_mismatch_prevention\Service\CurrencyValidationService;
use Drupal\commerce_currency_mismatch_prevention\Service\SettingsService;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_order\Entity\OrderItemInterface;
use Drupal\commerce_price\Exception\CurrencyMismatchException;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Tests\UnitTestCase;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;
use PHPUnit\Framework\Attributes\IgnoreDeprecations;
use PHPUnit\Framework\MockObject\MockObject;

/**
 * Tests the CartManagerDecorator service.
 */
#[Group('commerce_currency_mismatch_prevention')]
#[IgnoreDeprecations]
#[CoversClass(CartManagerDecorator::class)]
class CartManagerDecoratorTest extends UnitTestCase {

  /**
   * The cart manager decorator.
   *
   * @var \Drupal\commerce_currency_mismatch_prevention\CartManagerDecorator
   */
  protected CartManagerDecorator $cartManagerDecorator;

  /**
   * The wrapped cart manager.
   *
   * @var \Drupal\commerce_cart\CartManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected CartManagerInterface|MockObject $cartManager;

  /**
   * The currency validation service.
   *
   * @var \Drupal\commerce_currency_mismatch_prevention\Service\CurrencyValidationService|\PHPUnit\Framework\MockObject\MockObject
   */
  protected CurrencyValidationService|MockObject $validationService;

  /**
   * The settings service.
   *
   * @var \Drupal\commerce_currency_mismatch_prevention\Service\SettingsService|\PHPUnit\Framework\MockObject\MockObject
   */
  protected SettingsService|MockObject $settings;

  /**
   * The message service.
   *
   * @var \Drupal\commerce_currency_mismatch_prevention\Service\CurrencyMessageService|\PHPUnit\Framework\MockObject\MockObject
   */
  protected CurrencyMessageService|MockObject $messageService;

  /**
   * The logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected LoggerChannelInterface|MockObject $logger;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->cartManager = $this->createMock(CartManagerInterface::class);
    $this->validationService = $this->createMock(CurrencyValidationService::class);
    $this->settings = $this->createMock(SettingsService::class);
    $this->messageService = $this->createMock(CurrencyMessageService::class);
    $this->logger = $this->createMock(LoggerChannelInterface::class);

    $this->cartManagerDecorator = new CartManagerDecorator(
      $this->cartManager,
      $this->validationService,
      $this->messageService,
      $this->logger,
      $this->settings
    );
  }

  /**
   * Tests addEntity when module is disabled and exception occurs.
   */
  public function testAddEntityDisabledThrowsException(): void {
    $cart = $this->createMock(OrderInterface::class);
    $entity = $this->createMock(PurchasableEntityInterface::class);
    $exception = new CurrencyMismatchException('Currency mismatch');

    $validation = new CurrencyValidationResult(TRUE);
    $this->validationService->expects($this->once())
      ->method('validateEntity')
      ->with($cart, $entity)
      ->willReturn($validation);

    $this->settings->expects($this->once())
      ->method('getCartBehavior')
      ->willReturn('disabled');

    $this->cartManager->expects($this->once())
      ->method('addEntity')
      ->with($cart, $entity, '1', TRUE, TRUE)
      ->willThrowException($exception);

    $this->expectException(CurrencyMismatchException::class);
    $this->expectExceptionMessage('Currency mismatch');

    $this->cartManagerDecorator->addEntity($cart, $entity);
  }

  /**
   * Tests addEntity with remove_existing behavior.
   */
  public function testAddEntityRemoveExistingSuccess(): void {
    $cart = $this->createMock(OrderInterface::class);
    $entity = $this->createMock(PurchasableEntityInterface::class);
    $orderItem = $this->createMock(OrderItemInterface::class);

    $validation = new CurrencyValidationResult(FALSE, 'EUR', 'USD');
    $this->validationService->expects($this->once())
      ->method('validateEntity')
      ->with($cart, $entity)
      ->willReturn($validation);

    $this->settings->expects($this->once())
      ->method('getCartBehavior')
      ->willReturn('remove_existing');

    $this->messageService->expects($this->once())
      ->method('showCartEmptiedMessage')
      ->with($validation);

    $this->cartManager->expects($this->once())
      ->method('emptyCart')
      ->with($cart, FALSE);

    $this->cartManager->expects($this->once())
      ->method('addEntity')
      ->with($cart, $entity, '1', TRUE, TRUE)
      ->willReturn($orderItem);

    $result = $this->cartManagerDecorator->addEntity($cart, $entity);
    $this->assertSame($orderItem, $result);
  }

  /**
   * Tests addEntity with keep_current behavior.
   */
  public function testAddEntityKeepCurrentRejectsProduct(): void {
    $cart = $this->createMock(OrderInterface::class);
    $entity = $this->createMock(PurchasableEntityInterface::class);

    $validation = new CurrencyValidationResult(FALSE, 'EUR', 'USD');
    $this->validationService->expects($this->once())
      ->method('validateEntity')
      ->with($cart, $entity)
      ->willReturn($validation);

    $this->settings->expects($this->once())
      ->method('getCartBehavior')
      ->willReturn('keep_current');

    $this->messageService->expects($this->once())
      ->method('showProductRejectedMessage')
      ->with($validation);

    $this->cartManager->expects($this->never())
      ->method('addEntity');

    $result = $this->cartManagerDecorator->addEntity($cart, $entity);
    $this->assertNull($result);
  }

  /**
   * Tests addEntity with remove_existing behavior when exception occurs.
   */
  public function testAddEntityRemoveExistingAfterException(): void {
    $cart = $this->createMock(OrderInterface::class);
    $entity = $this->createMock(PurchasableEntityInterface::class);
    $orderItem = $this->createMock(OrderItemInterface::class);
    $exception = new CurrencyMismatchException('Currency mismatch');

    $validation = new CurrencyValidationResult(TRUE);
    $this->validationService->expects($this->once())
      ->method('validateEntity')
      ->with($cart, $entity)
      ->willReturn($validation);

    $this->settings->expects($this->once())
      ->method('getCartBehavior')
      ->willReturn('remove_existing');

    // Mock cart manager to throw exception first, then succeed.
    $callCount = 0;
    $this->cartManager->expects($this->exactly(2))
      ->method('addEntity')
      ->with($cart, $entity, '1', TRUE, TRUE)
      ->willReturnCallback(function () use (&$callCount, $exception, $orderItem) {
        $callCount++;
        if ($callCount === 1) {
          throw $exception;
        }
        return $orderItem;
      });

    $this->messageService->expects($this->once())
      ->method('showCartEmptiedMessage');

    $this->cartManager->expects($this->once())
      ->method('emptyCart')
      ->with($cart, FALSE);

    $result = $this->cartManagerDecorator->addEntity($cart, $entity);
    $this->assertSame($orderItem, $result);
  }

  /**
   * Tests addEntity with keep_current behavior when exception occurs.
   */
  public function testAddEntityKeepCurrentAfterException(): void {
    $cart = $this->createMock(OrderInterface::class);
    $entity = $this->createMock(PurchasableEntityInterface::class);
    $exception = new CurrencyMismatchException('Currency mismatch');

    $cart->expects($this->once())
      ->method('getItems')
      ->willReturn([]);

    $validation = new CurrencyValidationResult(TRUE);
    $this->validationService->expects($this->once())
      ->method('validateEntity')
      ->with($cart, $entity)
      ->willReturn($validation);

    $this->settings->expects($this->once())
      ->method('getCartBehavior')
      ->willReturn('keep_current');

    $this->cartManager->expects($this->once())
      ->method('addEntity')
      ->with($cart, $entity, '1', TRUE, TRUE)
      ->willThrowException($exception);

    $this->messageService->expects($this->once())
      ->method('showProductRejectedMessage');

    $result = $this->cartManagerDecorator->addEntity($cart, $entity);
    $this->assertNull($result);
  }

  /**
   * Tests addOrderItem when module is disabled and exception occurs.
   */
  public function testAddOrderItemDisabledThrowsException(): void {
    $cart = $this->createMock(OrderInterface::class);
    $orderItem = $this->createMock(OrderItemInterface::class);
    $exception = new CurrencyMismatchException('Currency mismatch');

    $validation = new CurrencyValidationResult(TRUE);
    $this->validationService->expects($this->once())
      ->method('validateOrderItem')
      ->with($cart, $orderItem)
      ->willReturn($validation);

    $this->settings->expects($this->once())
      ->method('getCartBehavior')
      ->willReturn('disabled');

    $this->cartManager->expects($this->once())
      ->method('addOrderItem')
      ->with($cart, $orderItem, TRUE, TRUE)
      ->willThrowException($exception);

    $this->expectException(CurrencyMismatchException::class);
    $this->expectExceptionMessage('Currency mismatch');

    $this->cartManagerDecorator->addOrderItem($cart, $orderItem);
  }

  /**
   * Tests addOrderItem with remove_existing behavior.
   */
  public function testAddOrderItemRemoveExistingSuccess(): void {
    $cart = $this->createMock(OrderInterface::class);
    $orderItem = $this->createMock(OrderItemInterface::class);

    $validation = new CurrencyValidationResult(FALSE, 'EUR', 'USD');
    $this->validationService->expects($this->once())
      ->method('validateOrderItem')
      ->with($cart, $orderItem)
      ->willReturn($validation);

    $this->settings->expects($this->once())
      ->method('getCartBehavior')
      ->willReturn('remove_existing');

    $this->messageService->expects($this->once())
      ->method('showCartEmptiedMessage')
      ->with($validation);

    $this->cartManager->expects($this->once())
      ->method('emptyCart')
      ->with($cart, FALSE);

    $this->cartManager->expects($this->once())
      ->method('addOrderItem')
      ->with($cart, $orderItem, TRUE, TRUE)
      ->willReturn($orderItem);

    $result = $this->cartManagerDecorator->addOrderItem($cart, $orderItem);
    $this->assertSame($orderItem, $result);
  }

  /**
   * Tests addOrderItem with keep_current behavior.
   */
  public function testAddOrderItemKeepCurrentRejectsProduct(): void {
    $cart = $this->createMock(OrderInterface::class);
    $orderItem = $this->createMock(OrderItemInterface::class);

    $validation = new CurrencyValidationResult(FALSE, 'EUR', 'USD');
    $this->validationService->expects($this->once())
      ->method('validateOrderItem')
      ->with($cart, $orderItem)
      ->willReturn($validation);

    $this->settings->expects($this->once())
      ->method('getCartBehavior')
      ->willReturn('keep_current');

    $this->messageService->expects($this->once())
      ->method('showProductRejectedMessage')
      ->with($validation);

    $this->cartManager->expects($this->never())
      ->method('addOrderItem');

    $result = $this->cartManagerDecorator->addOrderItem($cart, $orderItem);
    $this->assertSame($orderItem, $result);
  }

  /**
   * Tests successful addEntity when validation passes.
   */
  public function testAddEntitySuccessWhenValidationPasses(): void {
    $cart = $this->createMock(OrderInterface::class);
    $entity = $this->createMock(PurchasableEntityInterface::class);
    $orderItem = $this->createMock(OrderItemInterface::class);

    $validation = new CurrencyValidationResult(TRUE);
    $this->validationService->expects($this->once())
      ->method('validateEntity')
      ->with($cart, $entity)
      ->willReturn($validation);

    $this->cartManager->expects($this->once())
      ->method('addEntity')
      ->with($cart, $entity, '1', TRUE, TRUE)
      ->willReturn($orderItem);

    $this->messageService->expects($this->never())
      ->method('showCartEmptiedMessage');
    $this->messageService->expects($this->never())
      ->method('showProductRejectedMessage');

    $result = $this->cartManagerDecorator->addEntity($cart, $entity);
    $this->assertSame($orderItem, $result);
  }

  /**
   * Tests successful addOrderItem when validation passes.
   */
  public function testAddOrderItemSuccessWhenValidationPasses(): void {
    $cart = $this->createMock(OrderInterface::class);
    $orderItem = $this->createMock(OrderItemInterface::class);

    $validation = new CurrencyValidationResult(TRUE);
    $this->validationService->expects($this->once())
      ->method('validateOrderItem')
      ->with($cart, $orderItem)
      ->willReturn($validation);

    $this->cartManager->expects($this->once())
      ->method('addOrderItem')
      ->with($cart, $orderItem, TRUE, TRUE)
      ->willReturn($orderItem);

    $this->messageService->expects($this->never())
      ->method('showCartEmptiedMessage');
    $this->messageService->expects($this->never())
      ->method('showProductRejectedMessage');

    $result = $this->cartManagerDecorator->addOrderItem($cart, $orderItem);
    $this->assertSame($orderItem, $result);
  }

  /**
   * Tests that other methods are delegated properly.
   */
  public function testDelegatedMethods(): void {
    $cart = $this->createMock(OrderInterface::class);
    $entity = $this->createMock(PurchasableEntityInterface::class);
    $orderItem = $this->createMock(OrderItemInterface::class);

    $this->cartManager->expects($this->once())
      ->method('emptyCart')
      ->with($cart, TRUE);
    $this->cartManagerDecorator->emptyCart($cart, TRUE);

    $this->cartManager->expects($this->once())
      ->method('createOrderItem')
      ->with($entity, '2')
      ->willReturn($orderItem);
    $result = $this->cartManagerDecorator->createOrderItem($entity, '2');
    $this->assertSame($orderItem, $result);

    $this->cartManager->expects($this->once())
      ->method('updateOrderItem')
      ->with($cart, $orderItem, FALSE)
      ->willReturn($orderItem);
    $result = $this->cartManagerDecorator->updateOrderItem($cart, $orderItem, FALSE);
    $this->assertSame($orderItem, $result);

    $this->cartManager->expects($this->once())
      ->method('removeOrderItem')
      ->with($cart, $orderItem, TRUE);
    $this->cartManagerDecorator->removeOrderItem($cart, $orderItem, TRUE);
  }

}
