/**
 * @file
 * Defines behaviors for the CyberSource payment method form.
 */

/* global Flex */
(function (Drupal, drupalSettings, once) {
  /**
   * Attaches the commerceCyberSource behavior.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the commerceCyberSource behavior.
   *
   * @see Drupal.commerceCyberSource
   */
  Drupal.behaviors.commerceCyberSourceForm = {
    attach(context) {
      const cybersourceSettings = drupalSettings.commerceCyberSource;
      if (!cybersourceSettings) {
        return;
      }
      let microform;

      once(
        'commerceCyberSourceFlex',
        '.commerce-checkout-flow',
        context,
      ).forEach((checkoutForm) => {
        const clientLibrary = cybersourceSettings?.clientLibrary;
        const clientToken = cybersourceSettings?.clientToken;

        if (!clientLibrary || !clientToken) {
          return;
        }

        if (!document.querySelector(`script[src="${clientLibrary}"]`)) {
          const script = document.createElement('script');
          script.src = clientLibrary;
          script.setAttribute('crossorigin', 'anonymous');
          if (cybersourceSettings.clientLibraryIntegrity) {
            script.setAttribute(
              'integrity',
              cybersourceSettings.clientLibraryIntegrity,
            );
          }
          script.async = true;

          script.onload = () => {
            const flex = new Flex(clientToken);

            microform = flex.microform('card', {
              styles: {
                input: {
                  'font-size': '14px',
                  'font-family': 'Lucida Sans Unicode, Verdana, sans-serif',
                },
                ':disabled': {
                  cursor: 'not-allowed',
                },
                valid: {
                  color: '#3c763d',
                },
                invalid: {
                  color: '#a94442',
                },
              },
            });

            microform
              .createField('number', { placeholder: 'Enter card number' })
              .load('#cybersource-card-number');

            microform
              .createField('securityCode', { placeholder: '•••' })
              .load('#cybersource-card-cvv');
          };

          script.onerror = () => {
            console.error('Failed to load Flex SDK:', clientLibrary);
          };

          document.head.appendChild(script);
        }

        checkoutForm.addEventListener('submit', (event) => {
          if (!checkoutForm.dataset.canSubmit) {
            const cybersourceMonth =
              checkoutForm.querySelector('.cybersource-month');
            const cybersourceYear =
              checkoutForm.querySelector('.cybersource-year');
            if (!cybersourceMonth || !cybersourceYear) {
              return;
            }
            event.preventDefault();
            const submitButtons =
              checkoutForm.querySelectorAll('[type="submit"]');
            const lastButton = submitButtons[submitButtons.length - 1];
            lastButton.setAttribute('disabled', '');

            microform.createToken(
              {
                expirationMonth: cybersourceMonth.value,
                expirationYear: cybersourceYear.value,
              },
              function (err, token) {
                if (err) {
                  switch (err.reason) {
                    case 'CREATE_TOKEN_TIMEOUT':
                    case 'CREATE_TOKEN_NO_FIELDS_LOADED':
                    case 'CREATE_TOKEN_NO_FIELDS':
                    case 'CREATE_TOKEN_VALIDATION_PARAMS':
                    case 'CREATE_TOKEN_VALIDATION_FIELDS':
                    case 'CREATE_TOKEN_VALIDATION_SERVERSIDE':
                    case 'CREATE_TOKEN_UNABLE_TO_START':
                      console.error(err.message);
                      lastButton?.removeAttribute('disabled');
                      return;
                    default:
                      lastButton?.removeAttribute('disabled');
                      console.error('Unknown error');
                      return;
                  }
                }
                document.querySelector('.cybersource-token').value = token;
                checkoutForm.dataset.canSubmit = true;
                if (!checkoutForm.dataset.submitted) {
                  // Prevent double submissions.
                  checkoutForm.dataset.submitted = true;
                  checkoutForm.dataset.canSubmit = true;
                  checkoutForm.submit();
                }
              },
            );
          }
        });
      });
    },
  };
})(Drupal, drupalSettings, once);
