/**
 * @file
 * Defines behaviors for the CyberSource payment method form.
 */

/* global Flex */
(function (Drupal, drupalSettings, once) {
  /**
   * Theme function building the device data collection iframe.
   *
   * @param collectionUrl
   *   The Device data collection url
   * @param jwtToken
   *   The JWT access token.
   *
   * @returns {HTMLDivElement}
   */
  Drupal.theme.commerceCybersourceDeviceDataIframe = (
    collectionUrl,
    jwtToken,
  ) => {
    const cardinalCollectionIframe = document.createElement('iframe');
    cardinalCollectionIframe.setAttribute('id', 'cardinal-collection-iframe');
    cardinalCollectionIframe.setAttribute('name', 'cardinal-collection-iframe');
    cardinalCollectionIframe.setAttribute('height', '10');
    cardinalCollectionIframe.setAttribute('width', '10');
    cardinalCollectionIframe.setAttribute('style', 'display: none;');
    const cardinalCollectionForm = document.createElement('form');
    cardinalCollectionForm.setAttribute('id', 'cardinal-collection-form');
    cardinalCollectionForm.setAttribute('method', 'POST');
    cardinalCollectionForm.setAttribute('target', 'cardinal-collection-iframe');
    cardinalCollectionForm.setAttribute('action', collectionUrl);
    const cardinalCollectionHiddenInput = document.createElement('input');
    cardinalCollectionHiddenInput.setAttribute('type', 'hidden');
    cardinalCollectionHiddenInput.setAttribute('name', 'JWT');
    cardinalCollectionHiddenInput.setAttribute('value', jwtToken);
    cardinalCollectionForm.appendChild(cardinalCollectionHiddenInput);

    const container = document.createElement('div');
    container.appendChild(cardinalCollectionIframe);
    container.appendChild(cardinalCollectionForm);

    return container;
  };

  /**
   * Attaches the commerceCyberSource behavior.
   *
   * @type {Drupal~behavior}
   *
   * @prop {Drupal~behaviorAttach} attach
   *   Attaches the commerceCyberSource behavior.
   *
   * @see Drupal.commerceCyberSource
   */
  Drupal.behaviors.commerceCyberSourceForm = {
    attach(context) {
      const cybersourceSettings = drupalSettings.commerceCyberSource;
      if (!cybersourceSettings) {
        return;
      }

      function scrollToFlexError() {
        const el = document.getElementById('cybersource-errors');
        if (!el) {
          return;
        }
        el.hidden = false;
        el.scrollIntoView({ behavior: 'smooth', block: 'center' });
        el.focus({ preventScroll: true });
      }

      function showFlexError(message) {
        const el = document.getElementById('cybersource-errors');

        if (!el) {
          return;
        }

        el.textContent = message;
        el.hidden = false;
        scrollToFlexError();
      }

      function clearFlexError() {
        const el = document.getElementById('cybersource-errors');

        if (!el) {
          return;
        }

        el.textContent = '';
        el.hidden = true;
      }

      /**
       * Checks whether the given origin is a trusted origin.
       *
       * This is needed for the device data collection.
       *
       * @param origin
       *   The origin
       *
       * @returns boolean
       */
      function isTrustedCardinalOrigin(origin) {
        try {
          const url = new URL(origin);
          return (
            url.protocol === 'https:' &&
            (url.hostname === 'cardinalcommerce.com' ||
              url.hostname.endsWith('.cardinalcommerce.com'))
          );
        } catch {
          return false;
        }
      }

      async function payerAuthenticationSetup(paSetupUrl, token) {
        const response = await fetch(paSetupUrl, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({ token }),
        });

        return response.json();
      }

      let microform;
      once(
        'commerceCyberSourceFlex',
        '.commerce-checkout-flow',
        context,
      ).forEach((checkoutForm) => {
        const clientLibrary = cybersourceSettings?.clientLibrary;
        const clientToken = cybersourceSettings?.clientToken;

        if (!clientLibrary || !clientToken) {
          return;
        }

        function submitCheckoutForm(token) {
          checkoutForm.querySelector('.cybersource-token').value = token;
          if (!checkoutForm.dataset.submitted) {
            // Prevent double submissions.
            checkoutForm.dataset.submitted = true;
            checkoutForm.dataset.canSubmit = true;
            checkoutForm.submit();
          }
        }

        if (!document.querySelector(`script[src="${clientLibrary}"]`)) {
          const script = document.createElement('script');
          script.src = clientLibrary;
          script.setAttribute('crossorigin', 'anonymous');
          if (cybersourceSettings.clientLibraryIntegrity) {
            script.setAttribute(
              'integrity',
              cybersourceSettings.clientLibraryIntegrity,
            );
          }
          script.async = true;

          script.onload = () => {
            const flex = new Flex(clientToken);

            microform = flex.microform('card', {
              styles: {
                input: {
                  'font-size': '14px',
                  'font-family': 'Lucida Sans Unicode, Verdana, sans-serif',
                },
                ':disabled': {
                  cursor: 'not-allowed',
                },
                valid: {
                  color: '#3c763d',
                },
                invalid: {
                  color: '#a94442',
                },
              },
            });

            const cardNumberField = microform.createField('number', {
              placeholder: 'Enter card number',
            });
            cardNumberField.load('#cybersource-card-number');

            cardNumberField.on('focus', () => clearFlexError());

            const cvvField = microform.createField('securityCode', {
              placeholder: '•••',
            });
            cvvField.load('#cybersource-card-cvv');
            cvvField.on('focus', () => clearFlexError());
          };

          script.onerror = () => {
            console.error('Failed to load Flex SDK:', clientLibrary);
          };

          document.head.appendChild(script);
        }

        checkoutForm.addEventListener('submit', (event) => {
          if (!checkoutForm.dataset.canSubmit) {
            const cybersourceMonth =
              checkoutForm.querySelector('.cybersource-month');
            const cybersourceYear =
              checkoutForm.querySelector('.cybersource-year');
            if (!cybersourceMonth || !cybersourceYear) {
              return;
            }
            event.preventDefault();
            const submitButtons =
              checkoutForm.querySelectorAll('[type="submit"]');
            const lastButton = submitButtons[submitButtons.length - 1];
            lastButton.setAttribute('disabled', '');

            microform.createToken(
              {
                expirationMonth: cybersourceMonth.value,
                expirationYear: cybersourceYear.value,
              },
              async function (err, token) {
                if (err) {
                  switch (err.reason) {
                    case 'CREATE_TOKEN_TIMEOUT':
                    case 'CREATE_TOKEN_NO_FIELDS_LOADED':
                    case 'CREATE_TOKEN_NO_FIELDS':
                    case 'CREATE_TOKEN_VALIDATION_PARAMS':
                    case 'CREATE_TOKEN_VALIDATION_FIELDS':
                    case 'CREATE_TOKEN_VALIDATION_SERVERSIDE':
                    case 'CREATE_TOKEN_UNABLE_TO_START':
                      showFlexError(
                        err.message || Drupal.t('Invalid payment information.'),
                      );
                      lastButton?.removeAttribute('disabled');
                      return;
                    default:
                      lastButton?.removeAttribute('disabled');
                      showFlexError(
                        err.message || Drupal.t('Invalid payment information.'),
                      );
                      return;
                  }
                }

                // Payer authentication isn't enabled, submit the form.
                if (!cybersourceSettings?.payerAuthenticationSetupUrl) {
                  submitCheckoutForm(token);
                  return;
                }
                const paSetupData = await payerAuthenticationSetup(
                  cybersourceSettings.payerAuthenticationSetupUrl,
                  token,
                );

                // Attempt to submit the form, even in case the payer
                //  authentication setup call was unsuccessful.
                if (
                  !paSetupData?.deviceDataCollectionUrl ||
                  !paSetupData?.accessToken
                ) {
                  submitCheckoutForm(token);
                  return;
                }
                const iframeHtml =
                  Drupal.theme.commerceCybersourceDeviceDataIframe(
                    paSetupData.deviceDataCollectionUrl,
                    paSetupData.accessToken,
                  ).innerHTML;
                // Append the device data collection form after the checkout form.
                checkoutForm.insertAdjacentHTML('afterend', iframeHtml);

                // Create a flag to prevent double submission.
                let profileCompleted = false;

                // Set up a 10-second fallback timer, in case the device data
                // collection didn't complete, we should submit the form.
                const deviceDataFallbackTimeout = setTimeout(() => {
                  if (!profileCompleted) {
                    submitCheckoutForm(token);
                  }
                }, 10000);

                // Event listener listening for the device data completion.
                window.addEventListener('message', function (event) {
                  if (!isTrustedCardinalOrigin(event.origin)) {
                    return;
                  }
                  let data;
                  try {
                    data = JSON.parse(event.data);

                    // Device data collection is completed,
                    if (data.MessageType === 'profile.completed') {
                      // Mark as complete, so fallback won't run.
                      profileCompleted = true;
                      // Cancel the timeout.
                      clearTimeout(deviceDataFallbackTimeout);
                      submitCheckoutForm(token);
                    }
                  } catch (err) {
                    console.error('Invalid JSON in postMessage.', err);
                  }
                });

                const deviceDataForm = document.getElementById(
                  'cardinal-collection-form',
                );
                // Submit the device data collection form.
                if (deviceDataForm) {
                  deviceDataForm.submit();
                }
              },
            );
          }
        });
      });
    },
  };
})(Drupal, drupalSettings, once);
