<?php

namespace Drupal\commerce_cybersource\Plugin\Commerce\CheckoutPane;

use CyberSource\Api\PayerAuthenticationApi;
use CyberSource\Model\CheckPayerAuthEnrollmentRequest;
use CyberSource\Model\Riskv1authenticationsetupsClientReferenceInformation;
use CyberSource\Model\Riskv1authenticationsetupsTokenInformation;
use CyberSource\Model\Riskv1authenticationsOrderInformation;
use CyberSource\Model\Riskv1authenticationsOrderInformationAmountDetails;
use CyberSource\Model\Riskv1authenticationsOrderInformationBillTo;
use CyberSource\Model\Riskv1decisionsConsumerAuthenticationInformation;
use Drupal\commerce\Response\NeedsRedirectException;
use Drupal\commerce\Utility\Error;
use Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowInterface;
use Drupal\commerce_checkout\Plugin\Commerce\CheckoutPane\CheckoutPaneBase;
use Drupal\commerce_cybersource\Plugin\Commerce\PaymentGateway\FlexInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Url;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Payer authentication pane for Commerce Cybersource Flex.
 *
 * This checkout pane is required for payer authentication.
 * If the customer's card is not enrolled in 3DS then the form will submit as
 * normal. Otherwise, a modal will appear for the customer to authenticate and
 * approve of the charge.
 *
 * @CommerceCheckoutPane(
 *   id = "cybersource_flex_review",
 *   label = @Translation("Cybersource Flex review"),
 *   default_step = "review",
 *   wrapper_element = "container",
 * )
 */
class FlexReview extends CheckoutPaneBase {

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, ?CheckoutFlowInterface $checkout_flow = NULL) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition, $checkout_flow);
    $instance->logger = $container->get('logger.channel.commerce_cybersource');
    $instance->renderer = $container->get('renderer');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function isVisible(): bool {
    if ($this->order->get('payment_method')->isEmpty()) {
      return FALSE;
    }
    // No payer auth data is set on the order, we cannot continue.
    if (!$this->order->getData('cybersource_payer_auth')) {
      return FALSE;
    }
    /** @var \Drupal\commerce_payment\Entity\PaymentMethodInterface $payment_method */
    $payment_method = $this->order->get('payment_method')->entity;
    if ($payment_method->bundle() !== 'flex_credit_card') {
      return FALSE;
    }
    // Reusable payment methods aren't supposed to be presented a step-up
    // challenge.
    if ($payment_method->isReusable()) {
      return FALSE;
    }
    // Payer authentication isn't enabled, the pane shouldn't be visible.
    if (empty($payment_method->getPaymentGateway()->getPlugin()->getConfiguration()['enable_payer_authentication'])) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function buildPaneForm(array $pane_form, FormStateInterface $form_state, array &$complete_form) {
    $cybersource_payer_auth = $this->order->getData('cybersource_payer_auth', []);
    // The buildPaneForm() method is reinvoked after the Payer auth when the
    // pane is submitted.
    if (!empty($cybersource_payer_auth['authentication_status'])) {
      return $pane_form;
    }
    // For some reason, attaching the library to the $pane_form doesn't work
    // here...
    // Probably because the $pane_form doesn't return anything?
    $complete_form['#attached']['library'][] = 'commerce_cybersource/payer-authentication';
    $payment_method = $this->order->get('payment_method')->entity;
    assert($payment_method instanceof PaymentMethodInterface);
    $flex_plugin = $payment_method->getPaymentGateway()->getPlugin();
    assert($flex_plugin instanceof FlexInterface);
    $payment_gateway_configuration = $payment_method->getPaymentGateway()->getPlugin()->getConfiguration();

    if ($billing_profile = $payment_method->getBillingProfile()) {
      /** @var \Drupal\address\Plugin\Field\FieldType\AddressItem $address */
      $address = $billing_profile->get('address')->first();
      $billing_to_arr = [
        'firstName' => $address->getGivenName(),
        'lastName' => $address->getFamilyName(),
        'address1' => $address->getAddressLine1(),
        'address2' => $address->getAddressLine2(),
        'locality' => $address->getLocality(),
        'postalCode' => $address->getPostalCode(),
        'country' => $address->getCountryCode(),
        'administrativeArea' => $address->getAdministrativeArea(),
        'company' => [
          'name' => $address->getOrganization(),
        ],
      ];
    }
    $clientReferenceInformationArr = [
      'code' => $this->order->id(),
    ];
    $clientReferenceInformation = new Riskv1authenticationsetupsClientReferenceInformation($clientReferenceInformationArr);
    $orderInformationAmountDetailsArr = [
      'currency' => $this->order->getBalance()->getCurrencyCode(),
      'totalAmount' => $this->order->getBalance()->getNumber(),
    ];
    $orderInformationAmountDetails = new Riskv1authenticationsOrderInformationAmountDetails($orderInformationAmountDetailsArr);
    $orderInformationBillTo = new Riskv1authenticationsOrderInformationBillTo($billing_to_arr ?? []);

    $orderInformationArr = [
      'amountDetails' => $orderInformationAmountDetails,
      'billTo' => $orderInformationBillTo,
    ];
    $orderInformation = new Riskv1authenticationsOrderInformation($orderInformationArr);

    [$header, $payload, $signature] = explode('.', $payment_method->get('transient_token')->value);
    $payload = Json::decode(base64_decode($payload));
    $tokenInformationArr = [
      'jti' => $payload['jti'],
    ];
    $tokenInformation = new Riskv1authenticationsetupsTokenInformation($tokenInformationArr);

    $route_parameters = [
      'commerce_payment_gateway' => $payment_method->getPaymentGatewayId(),
    ];
    $options = [
      'query' => [
        'order_id' => $this->order->id(),
      ],
      'absolute' => TRUE,
    ];
    $cybersource_payer_auth_data = $this->order->getData('cybersource_payer_auth', []);
    $consumer_authentication = new Riskv1decisionsConsumerAuthenticationInformation([
      'referenceId' => $cybersource_payer_auth_data['referenceId'] ?? NULL,
      'returnUrl' => Url::fromRoute('commerce_payment.notify', $route_parameters, $options)->toString(),
    ]);

    $request_parameters = [
      'clientReferenceInformation' => $clientReferenceInformation,
      'orderInformation' => $orderInformation,
      'tokenInformation' => $tokenInformation,
      'consumerAuthenticationInformation' => $consumer_authentication,
    ];
    try {
      $api_client = $flex_plugin->getApiClient();
      $api_instance = new PayerAuthenticationApi($api_client);
      $request = new CheckPayerAuthEnrollmentRequest($request_parameters);

      // Log the API request if enabled.
      if (!empty($payment_gateway_configuration['log_api_calls'])) {
        $this->logger->notice('CheckPayerAuthEnrollmentRequest API request: <pre>@object</pre>', [
          '@object' => $request->__toString(),
        ]);
      }
      [$response, $status_code, $http_header] = $api_instance->checkPayerAuthEnrollment($request);

      if (!empty($payment_gateway_configuration['log_api_calls'])) {
        $this->logger->notice('CheckPayerAuthEnrollmentRequest API response - @status: <pre>@object</pre>', [
          '@status' => $response->getStatus() ?? 'UNKNOWN',
          '@object' => $response->__toString(),
        ]);
      }

      // The authentication failed, not really sure what we should do in this
      // case, display a generic error for now.
      if ($response->getStatus() === 'AUTHENTICATION_FAILED' || $status_code !== 201) {
        $this->logger->warning(sprintf('Payer authentication failed for order %s.<br/>Response: <pre>%s</pre>.', $this->order->id(), print_r($response, TRUE)));
        $this->redirectToPreviousStep();
        return $pane_form;
      }
      if ($response->getStatus() === 'AUTHENTICATION_SUCCESSFUL') {
        // If the review step should be skipped, redirect to the next checkout
        // step.
        if (!empty($payment_gateway_configuration['skip_review_step'])) {
          $previous_step_id = $this->checkoutFlow->getNextStepId($this->getStepId());
          $this->checkoutFlow->redirectToStep($previous_step_id);
        }

        return $pane_form;
      }

      // @todo check if other statuses are possible?
      if ($response->getStatus() === 'PENDING_AUTHENTICATION') {
        if (!empty($payment_gateway_configuration['auto_submit_review_step'])) {
          // Hide the actions buttons.
          $complete_form['actions']['#attributes']['class'][] = 'js-hide';
          $complete_form['#attributes']['class'][] = 'flex-autosubmit';
        }
        $consumer_auth_information = $response->getConsumerAuthenticationInformation();

        // The step-up iframe / form cannot be embedded within the checkout
        // form, so we append it to the form as a #suffix.
        $step_up_iframe = [
          '#theme' => 'commerce_cybersource_stepup_iframe',
          '#jwt_token' => $consumer_auth_information->getAccessToken(),
          '#stepup_url' => $consumer_auth_information->getStepUpUrl(),
        ];
        $complete_form['#suffix'] = $this->renderer->render($step_up_iframe);
      }
    }
    catch (\Exception $e) {
      if ($e instanceof NeedsRedirectException) {
        throw $e;
      }
      Error::logException($this->logger, $e);
      $this->redirectToPreviousStep();
      return $pane_form;
    }

    $cacheability = new CacheableMetadata();
    $cacheability->addCacheableDependency($this->order);
    $cacheability->setCacheMaxAge(0);
    $cacheability->applyTo($pane_form);

    return $pane_form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitPaneForm(array &$pane_form, FormStateInterface $form_state, array &$complete_form) {
    $cybersource_payer_auth = $this->order->getData('cybersource_payer_auth', [
      'authentication_successful' => FALSE,
    ]);
    if (empty($cybersource_payer_auth['authentication_successful'])) {
      $this->logger->warning(sprintf('Payer authentication failed for order %s.', $this->order->id()));
      $message = $this->t('We encountered an unexpected error processing your payment. Please try again using a different payment method or try again later.');
      $this->messenger()->addError($message);
      $this->redirectToPreviousStep();
    }
  }

  /**
   * Redirects to the previous checkout step.
   *
   * @throws \Drupal\commerce\Response\NeedsRedirectException
   */
  protected function redirectToPreviousStep(): void {
    $message = $this->t('We encountered an unexpected error processing your payment. Please try again using a different payment method or try again later.');
    $this->messenger()->addError($message);
    $previous_step_id = $this->checkoutFlow->getPreviousStepId($this->getStepId());
    $this->checkoutFlow->redirectToStep($previous_step_id);
  }

}
