/**
 * @file
 * Defines behaviors for the CyberSource payment method form.
 */

/* global Flex */
(function (Drupal, drupalSettings, once) {
  Drupal.theme.commerceCybersourceDeviceDataIframe = (
    collectionUrl,
    jwtToken,
  ) => {
    const iframe = document.createElement('iframe');
    iframe.id = 'cardinal-collection-iframe';
    iframe.name = 'cardinal-collection-iframe';
    iframe.height = 10;
    iframe.width = 10;
    iframe.style.display = 'none';

    const form = document.createElement('form');
    form.id = 'cardinal-collection-form';
    form.method = 'POST';
    form.target = 'cardinal-collection-iframe';
    form.action = collectionUrl;

    const input = document.createElement('input');
    input.type = 'hidden';
    input.name = 'JWT';
    input.value = jwtToken;
    form.appendChild(input);

    const container = document.createElement('div');
    container.appendChild(iframe);
    container.appendChild(form);

    return container;
  };

  Drupal.behaviors.commerceCyberSourceForm = {
    attach: (context) => {
      const cybersourceSettings = drupalSettings.commerceCyberSource;
      if (!cybersourceSettings) {
        return;
      }

      const scrollToFlexError = () => {
        const el = document.getElementById('cybersource-errors');
        if (!el) {
          return;
        }
        el.hidden = false;
        el.scrollIntoView({ behavior: 'smooth', block: 'center' });
        el.focus({ preventScroll: true });
      };

      const showFlexError = (message) => {
        const el = document.getElementById('cybersource-errors');
        if (!el) {
          return;
        }
        el.textContent = message;
        el.hidden = false;
        scrollToFlexError();
      };

      const clearFlexError = () => {
        const el = document.getElementById('cybersource-errors');
        if (!el) return;
        el.textContent = '';
        el.hidden = true;
      };

      const isTrustedCardinalOrigin = (origin) => {
        try {
          const url = new URL(origin);
          return (
            url.protocol === 'https:' &&
            (url.hostname === 'cardinalcommerce.com' ||
              url.hostname.endsWith('.cardinalcommerce.com'))
          );
        } catch {
          return false;
        }
      };

      const payerAuthenticationSetup = async (paSetupUrl, token) => {
        const response = await fetch(paSetupUrl, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ token }),
        });
        return response.json();
      };

      once(
        'commerceCyberSourceFlex',
        '.flex-credit-card-form',
        context,
      ).forEach((cardFormContainer) => {
        const clientLibrary = cybersourceSettings.clientLibrary;
        const clientToken = cybersourceSettings.clientToken;

        if (!clientLibrary || !clientToken) {
          return;
        }

        const checkoutForm = cardFormContainer.closest(
          '.commerce-checkout-flow',
        );
        if (!checkoutForm) {
          console.warn(
            'Could not find parent checkout form for card fields container',
          );
          return;
        }

        let microform;

        const submitCheckoutForm = (token) => {
          const tokenField = checkoutForm.querySelector('.cybersource-token');
          if (!tokenField) {
            return;
          }
          tokenField.value = token;

          if (!checkoutForm.dataset.cybersourceSubmitted) {
            checkoutForm.dataset.cybersourceSubmitted = '1';
            checkoutForm.dataset.cybersourceCanSubmit = '1';
            checkoutForm.submit();
          }
        };

        const initMicroform = () => {
          const flex = new Flex(clientToken);

          microform = flex.microform('card', {
            styles: {
              input: {
                'font-size': '14px',
                'font-family': 'Lucida Sans Unicode, Verdana, sans-serif',
              },
              ':disabled': { cursor: 'not-allowed' },
              valid: { color: '#3c763d' },
              invalid: { color: '#a94442' },
            },
          });

          const cardNumberField = microform.createField('number', {
            placeholder: 'Enter card number',
          });
          cardNumberField.load('#cybersource-card-number');
          cardNumberField.on('focus', () => clearFlexError());

          const cvvField = microform.createField('securityCode', {
            placeholder: '•••',
          });
          cvvField.load('#cybersource-card-cvv');
          cvvField.on('focus', () => clearFlexError());
        };

        if (!document.querySelector(`script[src="${clientLibrary}"]`)) {
          const script = document.createElement('script');
          script.src = clientLibrary;
          script.async = true;
          script.crossOrigin = 'anonymous';
          if (cybersourceSettings.clientLibraryIntegrity) {
            script.integrity = cybersourceSettings.clientLibraryIntegrity;
          }
          script.onload = initMicroform;
          script.onerror = () =>
            console.error('Failed to load Flex SDK:', clientLibrary);
          document.head.appendChild(script);
        } else {
          initMicroform();
        }

        // Attach a single submit listener to the checkout form
        if (!checkoutForm.dataset.cybersourceListenerAttached) {
          checkoutForm.addEventListener('submit', async (event) => {
            const cybersourceMonth =
              checkoutForm.querySelector('.cybersource-month');
            const cybersourceYear =
              checkoutForm.querySelector('.cybersource-year');

            // Only proceed if Flex fields exist (Flex is selected)
            if (!cybersourceMonth || !cybersourceYear) {
              return;
            }

            if (checkoutForm.dataset.cybersourceCanSubmit) {
              return;
            }

            event.preventDefault();

            const submitButtons =
              checkoutForm.querySelectorAll('[type="submit"]');
            const lastButton = submitButtons[submitButtons.length - 1];
            lastButton.setAttribute('disabled', '');

            microform.createToken(
              {
                expirationMonth: cybersourceMonth.value,
                expirationYear: cybersourceYear.value,
              },
              async (err, token) => {
                if (err) {
                  lastButton.removeAttribute('disabled');
                  showFlexError(
                    err.message || Drupal.t('Invalid payment information.'),
                  );
                  return;
                }

                if (!cybersourceSettings.payerAuthenticationSetupUrl) {
                  submitCheckoutForm(token);
                  return;
                }

                const paSetupData = await payerAuthenticationSetup(
                  cybersourceSettings.payerAuthenticationSetupUrl,
                  token,
                );

                if (
                  !paSetupData ||
                  !paSetupData.deviceDataCollectionUrl ||
                  !paSetupData.accessToken
                ) {
                  submitCheckoutForm(token);
                  return;
                }

                let deviceDataForm = document.getElementById(
                  'cardinal-collection-form',
                );
                if (!deviceDataForm) {
                  const iframeHtml =
                    Drupal.theme.commerceCybersourceDeviceDataIframe(
                      paSetupData.deviceDataCollectionUrl,
                      paSetupData.accessToken,
                    ).innerHTML;
                  checkoutForm.insertAdjacentHTML('afterend', iframeHtml);
                  deviceDataForm = document.getElementById(
                    'cardinal-collection-form',
                  );
                }

                let profileCompleted = false;
                const fallbackTimeout = setTimeout(() => {
                  if (!profileCompleted) {
                    submitCheckoutForm(token);
                  }
                }, 10000);

                const handleMessage = (event) => {
                  if (!isTrustedCardinalOrigin(event.origin)) {
                    return;
                  }

                  let data;
                  try {
                    data = JSON.parse(event.data);
                  } catch (err) {
                    console.warn('Failed to parse event data.', err);
                  }

                  if (data && data.MessageType === 'profile.completed') {
                    profileCompleted = true;
                    clearTimeout(fallbackTimeout);
                    submitCheckoutForm(token);
                    window.removeEventListener('message', handleMessage);
                  }
                };

                window.addEventListener('message', handleMessage);

                if (deviceDataForm) {
                  deviceDataForm.submit();
                }
              },
            );
          });

          checkoutForm.dataset.cybersourceListenerAttached = '1';
        }
      });
    },
  };
})(Drupal, drupalSettings, once);
