<?php

namespace Drupal\commerce_cybersource\Controller;

use CyberSource\Api\PayerAuthenticationApi;
use CyberSource\Model\PayerAuthSetupRequest;
use CyberSource\Model\Riskv1authenticationsetupsClientReferenceInformation;
use CyberSource\Model\Riskv1authenticationsetupsTokenInformation;
use Drupal\commerce_cybersource\Plugin\Commerce\PaymentGateway\FlexInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentGatewayInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Controller\ControllerBase;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides custom routes for the Payer authentication.
 */
class PayerAuthenticationController extends ControllerBase {

  /**
   * Constructs a new PayerAuthenticationController object.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger.
   */
  public function __construct(
    #[Autowire(service: 'logger.channel.commerce_cybersource')]
    protected LoggerInterface $logger,
  ) {
  }

  /**
   * Set up the Payer authentication for the given order.
   *
   * @param \Drupal\commerce_order\Entity\OrderInterface $commerce_order
   *   The order.
   * @param \Drupal\commerce_payment\Entity\PaymentGatewayInterface $commerce_payment_gateway
   *   The payment gateway.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   The JSON response.
   */
  public function setup(OrderInterface $commerce_order, PaymentGatewayInterface $commerce_payment_gateway, Request $request): JsonResponse {
    $payment_gateway_plugin = $commerce_payment_gateway->getPlugin();
    $configuration = $payment_gateway_plugin->getConfiguration();
    assert($payment_gateway_plugin instanceof FlexInterface);
    $api_client = $payment_gateway_plugin->getApiClient();
    $input = Json::decode($request->getContent());

    $parts = explode('.', $input['token']);
    $payload = Json::decode(base64_decode($parts[1]));
    try {
      $api_instance = new PayerAuthenticationApi($api_client);
      $request_object = new PayerAuthSetupRequest([
        'clientReferenceInformation' => new Riskv1authenticationsetupsClientReferenceInformation([
          'code' => $commerce_order->id(),
        ]),
        'tokenInformation' => new Riskv1authenticationsetupsTokenInformation([
          "transientToken" => $payload['jti'],
        ]),
      ]);
      if (!empty($configuration['log_api_calls'])) {
        $this->logger->notice('@class API request: <pre>@object</pre>', [
          '@class' => $request_object::class,
          '@object' => $request_object->__toString(),
        ]);
      }
      [$response, $status_code, $http_header] = $api_instance->payerAuthSetup($request_object);
      if (!empty($configuration['log_api_calls'])) {
        $this->logger->notice('@class API response - @status: <pre>@object</pre>', [
          '@class' => $request_object::class,
          '@status' => $response->getStatus() ?? 'UNKNOWN',
          '@object' => $response->__toString(),
        ]);
      }
      $consumer_auth_information = $response->getConsumerAuthenticationInformation();
      $commerce_order->setData('cybersource_payer_auth', [
        'referenceId' => $consumer_auth_information->getReferenceId(),
      ]);
      $commerce_order->setRefreshState(OrderInterface::REFRESH_SKIP);
      $commerce_order->save();

      return JsonResponse::fromJsonString((string) $response->getConsumerAuthenticationInformation());
    }
    catch (\Exception $e) {
      $commerce_order->unsetData('cybersource_payer_auth');
      $commerce_order->setRefreshState(OrderInterface::REFRESH_SKIP);
      $commerce_order->save();
      $this->logger->error($e->getMessage());
    }

    return new JsonResponse([]);
  }

}
