<?php

namespace Drupal\commerce_easypost\Plugin\Commerce\CheckoutPane;

use Drupal\commerce_checkout\Plugin\Commerce\CheckoutPane\CheckoutPaneBase;
use Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowInterface;
use Drupal\commerce_easypost\Service\EasyPostManager;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a pane for the customer to enter the recipient's name, phone number, and email address.
 *
 * @CommerceCheckoutPane(
 *   id = "commerce_easypost_phone",
 *   label = @Translation("Recipient Information"),
 *   wrapper_element = "fieldset",
 * )
 */
class CustomerPhoneNumber extends CheckoutPaneBase implements ContainerFactoryPluginInterface {

  /**
   * @var \Drupal\commerce_easypost\Service\EasyPostManager
   */
  protected $easyPostManager;

  /**
   * Constructs a new CheckoutPaneBase object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\commerce_checkout\Plugin\Commerce\CheckoutFlow\CheckoutFlowInterface $checkout_flow
   *   The parent checkout flow.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, CheckoutFlowInterface $checkout_flow, EntityTypeManagerInterface $entity_type_manager, EasyPostManager $easyPostManager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $checkout_flow, $entity_type_manager);

    $this->easyPostManager = $easyPostManager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, CheckoutFlowInterface $checkout_flow = NULL) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $checkout_flow,
      $container->get('entity_type.manager'),
      $container->get('commerce_easypost.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
        'collect_name' => TRUE,
        'collect_phone' => TRUE,
        'collect_email' => TRUE,
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['collect_name'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Collect recipient name'),
      '#default_value' => $this->configuration['collect_name'],
    ];
    $form['collect_phone'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Collect recipient phone number'),
      '#default_value' => $this->configuration['collect_phone'],
    ];
    $form['collect_email'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Collect recipient email address'),
      '#default_value' => $this->configuration['collect_email'],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['collect_name'] = !empty($values['collect_name']);
      $this->configuration['collect_phone'] = !empty($values['collect_phone']);
      $this->configuration['collect_email'] = !empty($values['collect_email']);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationSummary() {
    $parent_summary = parent::buildConfigurationSummary();
    if (!empty($this->configuration['collect_name'])) {
      $summary = $this->t('Collect recipient name: Yes') . '<br>';
    }
    else {
      $summary = $this->t('Collect recipient name: No') . '<br>';
    }
    if (!empty($this->configuration['collect_phone'])) {
      $summary .= $this->t('Collect recipient phone number: Yes') . '<br>';
    }
    else {
      $summary .= $this->t('Collect recipient phone number: No') . '<br>';
    }
    if (!empty($this->configuration['collect_email'])) {
      $summary .= $this->t('Collect recipient email address: Yes');
    }
    else {
      $summary .= $this->t('Collect recipient email address: No');
    }

    return $parent_summary ? implode('<br>', [$parent_summary, $summary]) : $summary;
  }

  /**
   * {@inheritdoc}
   */
  public function isVisible() {
    return TRUE;
  }

  public function buildPaneSummary() {
    $build = [
      'name' => [],
      'phone' => [],
      'email' => [],
    ];
    $name = $this->getCustomerName();
    if ($name !== NULL) {
      $build['name'] = [
        '#type' => 'inline_template',
        '#template' => '<div><span class="label">' . $this->t('Name') . '</span>: {{ name }}</div>',
        '#context' => ['name' => $name],
      ];
    }
    $phone = $this->getCustomerPhoneNumber();
    if ($phone !== NULL) {
      $build['phone'] = [
        '#type' => 'inline_template',
        '#template' => '<div><span class="label">' . $this->t('Phone Number') . '</span>: {{ phone }}</div>',
        '#context' => ['phone' => $phone],
      ];
    }
    $email = $this->getCustomerEmailAddress();
    if ($email !== NULL) {
      $build['email'] = [
        '#type' => 'inline_template',
        '#template' => '<div><span class="label">' . $this->t('Email Address') . '</span>: {{ email }}</div>',
        '#context' => ['email' => $email],
      ];
    }
    return $build;
  }

  /**
   * @return string|null
   */
  protected function getCustomerName() : ?string {
    if (empty($this->order->shipments)) {
      return NULL;
    }
    foreach ($this->order->shipments->referencedEntities() as $shipment) {
      /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment */
      $name = $shipment->get('easypost_name')->value;
      if ($name !== NULL) {
        return $name;
      }
    }
    return NULL;
  }

  /**
   * @return string|null
   */
  protected function getCustomerPhoneNumber() : ?string {
    if (empty($this->order->shipments)) {
      return NULL;
    }
    foreach ($this->order->shipments->referencedEntities() as $shipment) {
      /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment */
      $phone = $shipment->get('easypost_phone')->value;
      if ($phone !== NULL) {
        return $phone;
      }
    }
    return NULL;
  }

  /**
   * @return string|null
   */
  protected function getCustomerEmailAddress() : ?string {
    if (empty($this->order->shipments)) {
      return NULL;
    }
    foreach ($this->order->shipments->referencedEntities() as $shipment) {
      /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment */
      $email = $shipment->get('easypost_email')->value;
      if ($email !== NULL) {
        return $email;
      }
    }
    return NULL;
  }


  public function buildPaneForm(array $pane_form, FormStateInterface $form_state, array &$complete_form) {
    if ($this->configuration['collect_name']) {
      $pane_form['easypost_name'] = [
        '#type' => 'textfield',
        '#title' => t('Recipient Name'),
        '#required' => TRUE,
        '#default_value' => $this->getCustomerName(),
      ];
    }
    if ($this->configuration['collect_phone']) {
      $pane_form['easypost_phone'] = [
        '#type' => 'textfield',
        '#title' => t('Recipient Phone Number'),
        '#required' => TRUE,
        '#default_value' => $this->getCustomerPhoneNumber(),
      ];
    }
    if ($this->configuration['collect_email']) {
      $pane_form['easypost_email'] = [
        '#type' => 'email',
        '#title' => t('Recipient Email Address'),
        '#required' => TRUE,
        '#default_value' => $this->getCustomerEmailAddress(),
      ];
    }
    return $pane_form;
  }


  public function submitPaneForm(array &$pane_form, FormStateInterface $form_state, array &$complete_form) {
    $customer_name = $form_state->getValue('commerce_easypost_phone')['easypost_name'];
    $customer_phone = $form_state->getValue('commerce_easypost_phone')['easypost_phone'];
    $customer_email = $form_state->getValue('commerce_easypost_phone')['easypost_email'];
    // Update the shipments with the customer phone information
    foreach ($this->order->shipments->referencedEntities() as $shipment) {
      /** @var \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment */
      $plugin = $shipment->getShippingMethod()->getPlugin();
      if ($plugin->getPluginId() === 'easypost') {
        $shipment->set('easypost_name', $customer_name);
        $shipment->set('easypost_phone', $customer_phone);
        $shipment->set('easypost_email', $customer_email);
        $shipment->save();
      }

    }
  }


}
