<?php

namespace Drupal\commerce_easypost\Service;

use Drupal\commerce_easypost\Event\EasyPostEvents;
use Drupal\commerce_easypost\Event\EasyPostShipmentEvent;
use Drupal\commerce_price\Price;
use Drupal\commerce_price\RounderInterface;
use Drupal\commerce_shipping_label\ScheduledPickup;
use Drupal\commerce_shipping_label\ScheduledPickupRate;
use Drupal\commerce_shipping\Entity\ShipmentInterface;
use Drupal\commerce_shipping\ShippingRate;
use Drupal\commerce_shipping\ShippingService;
use Drupal\commerce_shipping_label\ShippingLabelManager;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\physical\Length;
use Drupal\physical\LengthUnit;
use Drupal\physical\Volume;
use Drupal\physical\VolumeUnit;
use EasyPost\EasyPostClient;
use EasyPost\Order;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

class EasyPostManager {

  use StringTranslationTrait;

  /**
   * The configuration array from a CommerceShippingMethod.
   *
   * @var array
   */
  protected $configuration;

  /**
   * @var string
   */
  protected $shippingMethodId;

  /**
   * The price rounder.
   *
   * @var \Drupal\commerce_price\RounderInterface
   */
  protected $rounder;

  /**
   * @var \Drupal\commerce_easypost\Service\EasyPostServiceDescriptions
   */
  protected $serviceDescriptions;

  /**
   * @var \EasyPost\Shipment[]
   */
  protected $easypostShipments;

  /**
   * @var \EasyPost\Order[]
   */
  protected $easypostOrders;

  /**
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * @var \Drupal\commerce_shipping_label\ShippingLabelManager
   */
  protected $shippingLabelManager;

  /**
   * @var \EasyPost\EasyPostClient
   */
  protected EasyPostClient $easyPostClient;

  /**
   * EasyPostManager constructor.
   *
   * @param \Drupal\commerce_price\RounderInterface $rounder
   */
  public function __construct(RounderInterface $rounder, EasyPostServiceDescriptions $serviceDescriptions, EventDispatcherInterface $eventDispatcher, ShippingLabelManager $shippingLabelManager) {
    $this->rounder = $rounder;
    $this->serviceDescriptions = $serviceDescriptions;
    $this->eventDispatcher = $eventDispatcher;
    $this->easypostShipments = [];
    $this->easypostOrders = [];
    $this->shippingLabelManager = $shippingLabelManager;

  }

  /**
   * @throws \EasyPost\Exception\General\MissingParameterException
   */
  public function setConfig(array $configuration, string $shippingMethodId) {
    $this->configuration = $configuration;
    $this->shippingMethodId = $shippingMethodId;
    $this->easyPostClient = new EasyPostClient($this->configuration['api_information']['api_key']);
  }


  /**
   * @param Order $easypost_order
   *
   * @return ShippingRate[]
   */
  public function getRates(\EasyPost\Order $easypost_order) : array {

    $rates = [];
    if (!empty($easypost_order->rates)) {
      $multiplier = (!empty($this->configuration['options']['rate_multiplier']))
        ? $this->configuration['options']['rate_multiplier']
        : 1.0;
      $round = !empty($this->configuration['options']['round'])
        ? $this->configuration['options']['round']
        : PHP_ROUND_HALF_UP;
      $enabled_services = $this->configuration['enabled_services'];
      foreach ($easypost_order->rates as $rate) {
        if (!empty($enabled_services[$rate->carrier]['enabled'][$rate->service])) {
          $service = new ShippingService(
            $this->serviceDescriptions->getServiceId($rate->carrier, $rate->service),
            $this->serviceDescriptions->getServiceName($rate->carrier, $rate->service)
          );
          $price = new Price((string) $rate->rate, $rate->currency);
          if ($multiplier != 1) {
            $price = $price->multiply((string) $multiplier);
          }
          $price = $this->rounder->round($price, $round);

          $commerce_rate = new ShippingRate([
            'shipping_method_id' => $this->shippingMethodId,
            'service' => $service,
            'amount' => $price,
          ]);
          $data = $commerce_rate->getData();
          $data['easypost_rate'] = $rate;
          $commerce_rate->setData($data);
          $rates[] = $commerce_rate;
        }
      }
    }
    return $rates;
  }

  public function buyShipment(ShipmentInterface $shipment) : \EasyPost\Order {
    $easypost_order = $this->createEasyPostOrder($shipment);
    $rates = $this->getRates($easypost_order);
    foreach ($rates as $rate) {
      if ($rate->getService()->getId() === $shipment->getShippingService()) {
        $data = $rate->getData();
        if (!empty($data['easypost_rate'])) {
          try {
            $easypost_order = $this->easyPostClient->order->buy(
              $easypost_order->id,
              $data['easypost_rate']
            );
          }
          catch (\Error $error) {
            \Drupal::messenger()->addError($error->getMessage());
          }
        }
      }
    }
    return $easypost_order;
  }

  public function createEasyPostOrder(ShipmentInterface $shipment) : ?\EasyPost\Order {
    if ($shipment->getShippingProfile()->address->isEmpty()) {
      return NULL;
    }

    $easypost_shipments = [];

    /** @var \CommerceGuys\Addressing\AddressInterface $address */
    $address = $shipment->getOrder()->getStore()->getAddress();
    $from_address = $this->easyPostClient->address->create([
      "company" => $shipment->getOrder()->getStore()->getName(),
      "street1" => $address->getAddressLine1(),
      "street2" => $address->getAddressLine2(),
      "city"    => $address->getLocality(),
      "state"   => $address->getAdministrativeArea(),
      "zip"     => $address->getPostalCode(),
      "country" => $address->getCountryCode(),
      'phone' => $this->configuration['options']['sender_phone'],
      'federal_tax_id' => $this->configuration['customs']['tax_id']
    ]);
    /** @var \CommerceGuys\Addressing\AddressInterface $address */
    $address = $shipment->getShippingProfile()->get('address')->first();
    $to_address = $this->easyPostClient->address->create([
      "company" => $address->getOrganization(),
      "street1" => $address->getAddressLine1(),
      "street2" => $address->getAddressLine2(),
      "city"    => $address->getLocality(),
      "state"   => $address->getAdministrativeArea(),
      "zip"     => $address->getPostalCode(),
      "country" => $address->getCountryCode(),
      'verify' => ['delivery', 'zip4'],
      'name' => $shipment->get('easypost_name')->value,
      'phone' => $shipment->get('easypost_phone')->value,
      'email' => $shipment->get('easypost_email')->value,
    ]);

    $package_count = static::calculatePackageCount($shipment);
    if ($package_count === FALSE) {
      $package_count = 1;
    }
    $package_weight = ($shipment->getWeight()->convert('oz')->getNumber()) / $package_count;

    for ($i = 0; $i < $package_count; $i++) {
      $easypost_shipments[] = [
        'parcel' => [
          "length" => $shipment->getPackageType()->getLength()->convert('in')->getNumber(),
          "height" => $shipment->getPackageType()->getHeight()->convert('in')->getNumber(),
          "width" => $shipment->getPackageType()->getWidth()->convert('in')->getNumber(),
          "weight" => $package_weight,
        ],
      ];
    }
    $options = [
      'dropoff_type' => $this->configuration['options']['dropoff_type'],
      'incoterm' => $this->configuration['options']['incoterm'],
    ];
    if (!empty($this->configuration['options']['include_order_number']) && !empty($shipment->getOrder()->getPlacedTime())) {
      $options['invoice_number'] = $shipment->getOrder()->getOrderNumber();
      $options['print_custom_1'] = $shipment->getOrder()->getOrderNumber();
      $options['print_custom_1_code'] = 'IK';
    }
    $options = array_filter($options);

    /** @var \CommerceGuys\Addressing\AddressInterface $billing_address */
    if ($shipment->getOrder()->getBillingProfile() !== NULL) {
      $billing_address = $shipment->getOrder()->getBillingProfile()->get('address')->first();
    }
    $store_address = $shipment->getOrder()->getStore()->getAddress();

    $carrier_account_number = $this->getCarrierAccountNumber($shipment);
    if (!empty($carrier_account_number) && !empty($billing_address)) {
      $options['payment'] = [
        'type' => 'RECEIVER',
        'account' => $carrier_account_number,
        'postal_code' => $billing_address->getPostalCode(),
      ];
    }
    if ($this->isInternational($shipment)) {
      $customs_items = [];
      $total_items = 0;
      foreach ($shipment->getItems() as $item) {
        $total_items += (float) $item->getQuantity();
      }
      $weight_per_item = ((float) $shipment->getWeight()->convert('oz')->getNumber()) / $total_items;
      foreach ($shipment->getItems() as $item) {
        $customs_item_info = [
          'quantity' => (float) $item->getQuantity(),
          'weight' => $weight_per_item * $item->getQuantity(),
          'value' => $item->getDeclaredValue()->getNumber(),
          'currency' => $item->getDeclaredValue()->getCurrencyCode(),
          'origin_country' => $store_address->getCountryCode(),
        ];
        $customs_item_info['description'] = !empty($this->configuration['customs']['description']) ? $this->configuration['customs']['description'] : $this->t('Merchandise');
        if (!empty($this->configuration['customs']['hs_tariff_number'])) {
          $customs_item_info['hs_tariff_number'] = $this->configuration['customs']['hs_tariff_number'];
        }
        $customs_items[] = $customs_item_info;
      }
      $customs_info = [
        'contents_type' => 'merchandise',
        'customs_items' => $customs_items,
        'customs_certify' => true,
        'customs_signer' => !empty($this->configuration['customs']['customs_signer']) ? $this->configuration['customs']['customs_signer'] : '',
        'restriction_type' => 'none',
      ];
      if ($shipment->hasField('easypost_aes_itn') && !$shipment->get('easypost_aes_itn')->isEmpty()) {
        $customs_info['eel_pfc'] = $shipment->get('easypost_aes_itn')->value;
      }
      else {
        $customs_info['eel_pfc'] = 'NOEEI 30.37(a)';
      }
    }
    $values = [
      "to_address"   => $to_address,
      "from_address" => $from_address,
      "shipments"       => $easypost_shipments,
      'options'      => $options,
    ];
    if (isset($customs_info)) {
      $values['customs_info'] = $customs_info;
    }

    $event = new EasyPostShipmentEvent($shipment, $values);
    $this->eventDispatcher->dispatch($event, EasyPostEvents::SHIPMENT_PRE_CREATE);
    $values = $event->getValues();
    return $this->easyPostClient->order->create($values);
  }

  public static function calculatePackageCount(ShipmentInterface $shipment) {
    $package_type = $shipment->getPackageType();
    $package_volume = static::getPackageVolume($package_type->getLength(), $package_type->getWidth(), $package_type->getHeight());
    $items_volume = static::getPackageTotalVolume($shipment->getItems(), $package_volume->getUnit());
    if ($package_volume->getNumber() != 0) {
      return ceil($items_volume->getNumber() / $package_volume->getNumber());
    }
    return FALSE;
  }

  protected static function getPackageTotalVolume(array $shipment_items, ?string $volume_unit = NULL): Volume {
    if ($volume_unit === NULL) {
      $volume_unit = VolumeUnit::CUBIC_CENTIMETER;
    }
    switch ($volume_unit) {
      case VolumeUnit::CUBIC_CENTIMETER:
        $linear_unit = LengthUnit::CENTIMETER;
        break;

      case VolumeUnit::CUBIC_INCH:
        $linear_unit = LengthUnit::INCH;
        break;

      default:
        throw new \RuntimeException('Invalid Units');
    }
    $order_item_storage = \Drupal::entityTypeManager()->getStorage('commerce_order_item');
    $order_item_ids = [];
    foreach ($shipment_items as $shipment_item) {
      $order_item_ids[] = $shipment_item->getOrderItemId();
    }

    $order_items = $order_item_storage->loadMultiple($order_item_ids);

    $total_volume = 0;
    foreach ($order_items as $order_item) {
      /** @var \Drupal\commerce_order\Entity\OrderItem $order_item */
      /** @var \Drupal\commerce_product\Entity\ProductVariationInterface $purchased_entity */
      $purchased_entity = $order_item->getPurchasedEntity();

      if ($purchased_entity->hasField('dimensions') && !$purchased_entity->get('dimensions')->isEmpty()) {
        /** @var \Drupal\physical\Plugin\Field\FieldType\DimensionsItem $dimensions */
        $dimensions = $purchased_entity->get('dimensions')->first();

        $volume = $dimensions
          ->getHeight()
          ->convert($linear_unit)
          ->multiply($dimensions->getWidth()->convert($linear_unit)->getNumber())
          ->multiply($dimensions->getLength()->convert($linear_unit)->getNumber())
          ->getNumber();

        $total_volume += (float) $volume * $order_item->getQuantity();
      }
    }

    return new Volume((string) $total_volume, $volume_unit);
  }

  public static function getPackageVolume(Length $length, Length $width, Length $height): Volume {
    // Assume all units are the same.
    $units = $length->getUnit();
    $volume_value = $length->getNumber() * $width->getNumber() * $height->getNumber();
    switch ($units) {
      case LengthUnit::CENTIMETER:
        $volume_units = VolumeUnit::CUBIC_CENTIMETER;
        break;

      case LengthUnit::INCH:
        $volume_units = VolumeUnit::CUBIC_INCH;
        break;

      default:
        throw new \RuntimeException('Invalid Dimensions');
    }
    return new Volume((string) $volume_value, $volume_units);
  }

  public function getTrackingUrl(ShipmentInterface $shipment) {
    $remote_id = $this->shippingLabelManager->getRemoteShipmentId($shipment);
    if (!empty($remote_id)) {
      try {
        $easypost_order = $this->getEasyPostOrder($remote_id);
        if (!empty($easypost_order) && !empty($easypost_order->shipments)) {
          if (!empty($easypost_order->shipments[0]->tracker->public_url)) {
            return Url::fromUri($easypost_order->shipments[0]->tracker->public_url);
          }
        }

      }
      catch (\Error $e) {
        return NULL;
      }
    }
    return NULL;
  }

  /**
   * @param \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment
   *
   * @return \Drupal\commerce_price\Price
   */
  protected function getTotalItemPrice(ShipmentInterface $shipment) : Price {
    $total = new Price(0, 'USD');
    foreach ($shipment->getOrder()->getItems() as $item) {
      $total = $total->add($item->getAdjustedUnitPrice());
    }
    return $total;
  }

  /**
   * @param \Drupal\commerce_shipping\Entity\ShipmentInterface $shipment
   *
   * @return bool
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function isInternational(ShipmentInterface $shipment) {
    $shipping_address = NULL;
    if ($shipment->getShippingProfile() !== NULL) {
      $shipping_address = $shipment->getShippingProfile()->get('address')->first();
    }
    $store_address = $shipment->getOrder()->getStore()->getAddress();
    return $shipping_address !== NULL && ($shipping_address->getCountryCode() !== $store_address->getCountryCode());
  }

  /**
   * @return string|null
   */
  public function getCarrierAccountNumber(ShipmentInterface $shipment) : ?string {
    if (!empty($shipment->get('easypost_carrier_account_number')->value)) {
      return preg_replace('/\D/', '', $shipment->get('easypost_carrier_account_number')->value);
    }
    return '';
  }

  /**
   * @param string $easypost_order_id
   * @param bool $reset
   * @return Order|null
   */
  public function getEasyPostOrder(string $easypost_order_id, $reset = FALSE) : ?\EasyPost\Order {
    if ($reset || !isset($this->easypostOrders[$easypost_order_id])) {
      try {
        $this->easypostOrders[$easypost_order_id] = $this->easyPostClient->order->retrieve($easypost_order_id);
      }
      catch (\Exception $e) {
        return NULL;
      }
    }
    return $this->easypostOrders[$easypost_order_id];
  }


  public function getPickupRates(ScheduledPickup $pickup) : array {
    $pickup_rates = [];
    $shipment = $pickup->getShipment();
    if (empty($shipment->get('shipment_label_remote_id')->value)) {
      return $pickup_rates;
    }
    $easypost_order = $this->easyPostClient->order->retrieve($shipment->get('shipment_label_remote_id')->value);

    if (!empty($easypost_order->shipments)) {
      $values = [
        'shipment' => $easypost_order->shipments[0],
        'is_account_address' => $pickup->getDataValue('is_account_address'),
        'min_datetime' => $pickup->getDataValue('min_datetime'),
      ];

      $shipping_rate = NULL;

      foreach ($this->getRates($easypost_order) as $rate) {
        if ($rate->getService()->getId() === $shipment->getShippingService()) {
          $shipping_rate = $rate;
        }
      }
      if ($shipping_rate !== NULL) {
        $data = $shipping_rate->getData();
        $values['carrier_accounts'] = ['id' => $data['easypost_rate']->carrier_account_id];
      }


      if ($pickup->getDataValue('instructions') !== NULL) {
        $values['instructions'] = $pickup->getDataValue('instructions');
      }
      if ($pickup->getDataValue('max_datetime') !== NULL) {
        $values['max_datetime'] = $pickup->getDataValue('max_datetime');
      }
      if ($pickup->getDataValue('address') !== NULL) {
        $address = $pickup->getDataValue('address');
        $easypost_address = $this->easyPostClient->address->create([
          "company" => $shipment->getOrder()->getStore()->getName(),
          "street1" => $address['address_line1'],
          "street2" => $address['address_line2'],
          "city"    => $address['locality'],
          "state"   => $address['administrative_area'],
          "zip"     => $address['postal_code'],
          "country" => $address['country_code'],
          'phone' => $this->configuration['options']['sender_phone'],
        ]);
        $values['address'] = $easypost_address;
      }

      $easypost_pickup = $this->easyPostClient->pickup->create($values);
      $pickup->setRemoteId($easypost_pickup->id);
      if (!empty($easypost_pickup->pickup_rates)) {
        foreach ($easypost_pickup->pickup_rates as $easypost_pickup_rate) {
          $pickup_rate = new ScheduledPickupRate();
          $pickup_rate->setDescription($easypost_pickup_rate->carrier . ' ' . $easypost_pickup_rate->service)
            ->setPrice(new Price($easypost_pickup_rate->rate, $easypost_pickup_rate->currency))
            ->setRemoteId($easypost_pickup_rate->id)
            ->setDataValue('easypost_rate', $easypost_pickup_rate);
          $pickup_rates[] = $pickup_rate;
        }
      }
    }
    return $pickup_rates;
  }

  public function schedulePickup(ScheduledPickup $pickup, ScheduledPickupRate $selectedRate) {
    $easypost_pickup = $this->easyPostClient->pickup->retrieve($pickup->getRemoteId());
    $easypost_pickup = $this->easyPostClient->pickup->buy(
      $easypost_pickup->id,
      ['rate' => $selectedRate->getDataValue('easypost_rate'),]
    );
    if (!empty($easypost_pickup->confirmation)) {
      $pickup->setConfirmationNumber($easypost_pickup->confirmation);
    }
    if (!empty($easypost_pickup->status) && $easypost_pickup->status !== 'unknown') {
      $pickup->setStatus($easypost_pickup->status);
    }
    $pickup->setRemoteId($easypost_pickup->id);
    return $pickup;
  }

  public function cancelPickup(string $pickup_remote_id) : bool {
    $easypost_pickup = $this->easyPostClient->pickup->retrieve($pickup_remote_id);
    try {
      $easypost_pickup = $this->easyPostClient->pickup->cancel($pickup_remote_id);
      return TRUE;
    }
    catch (\Exception | \Error $e) {
      \Drupal::messenger()->addError($e->getMessage());
      return FALSE;
    }
  }

  /**
   * @return \EasyPost\EasyPostClient|null
   */
  public function getEasyPostClient() : ?\EasyPost\EasyPostClient {
    return $this->easyPostClient;
  }

}
