<?php

namespace Drupal\commerce_field_per_store;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\EntityReferenceFieldItemList;
use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Form\FormStateInterface;

/**
 * Defines an item list class for entity per store fields.
 */
class CommerceEntityReferencePerStoreItemList extends EntityReferenceFieldItemList implements EntityReferenceFieldItemListInterface {

  /**
   * {@inheritdoc}
   */
  public static function processDefaultValue($default_value, FieldableEntityInterface $entity, FieldDefinitionInterface $definition) {
    $default_value = parent::processDefaultValue($default_value, $entity, $definition);

    if ($default_value) {
      // Convert UUIDs to numeric IDs.
      $uuids = [];
      foreach ($default_value as $delta => $properties) {
        if (isset($properties['target_uuid'])) {
          $uuids[$delta] = $properties['target_uuid'];
        }
      }
      if ($uuids) {
        $target_type = $definition->getSetting('target_type');
        $entity_ids = \Drupal::entityQuery($target_type)
          ->condition('uuid', $uuids, 'IN')
          ->accessCheck(TRUE)
          ->execute();
        $entities = \Drupal::entityTypeManager()
          ->getStorage($target_type)
          ->loadMultiple($entity_ids);

        $entity_uuids = [];
        foreach ($entities as $id => $entity) {
          $entity_uuids[$entity->uuid()] = $id;
        }
        foreach ($uuids as $delta => $uuid) {
          $default_value[$delta]['target_id'] = $entity_uuids[$uuid] ?? NULL;
          unset($default_value[$delta]['target_uuid']);
        }
      }
    }

    return $default_value;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultValuesFormSubmit(array $element, array &$form, FormStateInterface $form_state) {
    $default_value = parent::defaultValuesFormSubmit($element, $form, $form_state);

    // Convert numeric IDs to UUIDs to ensure config deployability.
    $ids = [];
    foreach ($default_value as $delta => $properties) {
      $ids[] = $properties['target_revision_id'];
    }

    $entities = [];
    foreach ($ids as $id) {
      $entities[$id] = \Drupal::entityTypeManager()
        ->getStorage($this->getSetting('target_type'))
        ->loadRevision($id);
    }

    foreach ($default_value as $delta => $properties) {
      if (!empty($entities[$properties['target_revision_id']])) {
        $default_value[$delta] = [
          'enabled' => $properties['enabled'],
          'commerce_store' => $properties['commerce_store'],
          'target_uuid' => $entities[$properties['target_revision_id']]->uuid(),
          'target_revision_id' => $properties['target_revision_id'],
        ];
      }
    }
    return $default_value;
  }

  /**
   * {@inheritdoc}
   */
  public function hasAffectingChanges(FieldItemListInterface $original_items, $langcode) {
    // If there are fewer items, then it is a change.
    if (count($this) < count($original_items)) {
      return TRUE;
    }

    $original_items_by_store = self::itemsByStore($original_items);
    $items_by_store = self::itemsByStore($this);

    foreach ($items_by_store as $store_id => $item) {
      // If the original items do not have this store, then it is a change.
      if (!isset($original_items_by_store[$store_id])) {
        return TRUE;
      }

      // If the original items have this store, but the item is not the same,
      $original_item = $original_items_by_store[$store_id];
      if ($item->target_id != $original_item->target_id || $item->enabled != $original_item->enabled) {
        return TRUE;
      }

      // If it is the same entity, only consider it as having affecting changes
      // if the target entity itself has changes.
      if ($item->entity && $item->entity->hasTranslation($langcode) && $item->entity->getTranslation($langcode)->hasTranslationChanges()) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Makes list of items keyed by store.
   */
  protected static function itemsByStore(\ArrayAccess $items): array {
    $items_by_store = [];
    foreach ($items as $item) {
      $items_by_store[$item->commerce_store] = $item;
    }
    return $items_by_store;
  }

}
