<?php

namespace Drupal\commerce_field_per_store\Plugin\Field\FieldType;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;
use Drupal\Core\TypedData\DataReferenceTargetDefinition;
use Drupal\entity_reference_revisions\Plugin\Field\FieldType\EntityReferenceRevisionsItem;

/**
 * Field plugin for field per store.
 *
 * @FieldType(
 *   id = "commerce_entity_reference_per_store",
 *   label = @Translation("Field per store"),
 *   description = @Translation("Stores a revisionable entity reference with store scoping and an enable flag."),
 *   category = @Translation("Reference"),
 *   class = "\Drupal\commerce_field_per_store\Plugin\Field\FieldType\CommerceEntityReferencePerStoreItem",
 *   list_class = "\Drupal\commerce_field_per_store\CommerceEntityReferencePerStoreItemList",
 *   default_widget = "commerce_entity_reference_per_store_widget",
 *   default_formatter = "commerce_entity_reference_per_store_formatter"
 * )
 */
class CommerceEntityReferencePerStoreItem extends EntityReferenceRevisionsItem {

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition): array {
    $properties = parent::propertyDefinitions($field_definition);

    $properties['commerce_store'] = DataReferenceTargetDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Commerce store ID'))
      ->setDescription(t('The referenced commerce store ID.'))
      ->setSetting('unsigned', TRUE)
      ->setRequired(TRUE);

    $properties['enabled'] = DataDefinition::create('boolean')
      ->setLabel(t('Enabled'))
      ->setDescription(t('Whether this item is enabled for the given store.'));

    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition): array {
    $schema = parent::schema($field_definition);

    $schema['columns']['commerce_store'] = [
      'type' => 'int',
      'unsigned' => TRUE,
      'not null' => FALSE,
    ];
    $schema['columns']['enabled'] = [
      'type' => 'int',
      'size' => 'tiny',
      'not null' => TRUE,
      'default' => 0,
    ];

    return $schema;
  }

  /**
   * {@inheritdoc}
   */
  public function isEmpty(): bool {
    return empty($this->commerce_store);
  }

  /**
   * {@inheritdoc}
   */
  public function getValue() {
    if (!isset($this->entity)) {
      $settings = $this->definition['settings'];
      $target_type = $settings['target_type'];
      $handler_settings = $settings['handler_settings'];
      $bundle = reset($handler_settings['target_bundles']);
      $this->entity = \Drupal::entityTypeManager()
        ->getStorage($target_type)
        ->create(['type' => $bundle]);
      $this->target_id = NULL;
      $this->target_revision_id = NULL;
    }
    return parent::getValue();
  }

}
