<?php

namespace Drupal\commerce_field_per_store\Plugin\Field\FieldWidget;

use Drupal\Component\Utility\NestedArray;
use Drupal\Core\Entity\EntityFormBuilderInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\WidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'commerce_entity_reference_per_store_widget'
 * widget.
 *
 * @FieldWidget(
 *   id = "commerce_entity_reference_per_store_widget",
 *   label = @Translation("Field per store"),
 *   field_types = {
 *     "commerce_entity_reference_per_store"
 *   }
 * )
 */
class CommerceEntityReferencePerStoreWidget extends WidgetBase implements ContainerFactoryPluginInterface {

  /**
   * The entity form builder service.
   *
   * @var \Drupal\Core\Entity\EntityFormBuilderInterface
   */
  protected $entityFormBuilder;

  /**
   * Constructs a CommerceEntityReferencePerStoreWidget object.
   *
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The field definition.
   * @param array $settings
   *   The widget settings.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Entity\EntityFormBuilderInterface $entity_form_builder
   *   The entity form builder.
   */
  public function __construct($plugin_id, $plugin_definition, FieldDefinitionInterface $field_definition, array $settings, array $third_party_settings, EntityFormBuilderInterface $entity_form_builder) {
    parent::__construct($plugin_id, $plugin_definition, $field_definition, $settings, $third_party_settings);
    $this->entityFormBuilder = $entity_form_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('entity.form_builder')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formMultipleElements(FieldItemListInterface $items, array &$form, FormStateInterface $form_state): array {
    // Build a mapping of existing items: commerce_store => delta.
    $store_delta_mapping = [];
    foreach ($items as $delta => $item) {
      if (!empty($item->commerce_store)) {
        $store_delta_mapping[$item->commerce_store] = $delta;
      }
    }

    // Load all commerce store entities.
    $store_storage = \Drupal::entityTypeManager()->getStorage('commerce_store');
    $stores = $store_storage->loadMultiple();

    $elements = [
      '#type' => 'details',
      '#open' => TRUE,
      '#title' => $this->fieldDefinition->getLabel(),
      '#title_display' => 'before',
    ];

    $base_element = [];
    foreach ($stores as $store) {
      $commerce_store = $store->id();

      if (isset($store_delta_mapping[$commerce_store])) {
        // Use the existing delta.
        $delta = $store_delta_mapping[$commerce_store];
      }
      else {
        // No existing item found, create a new one.
        $delta = count($items);
        $items->appendItem([
          'enabled' => 0,
          'commerce_store' => $commerce_store,
          'entity' => NULL,
        ]);
      }

      // Build the widget element for this delta.
      $elements[$delta] = $this->formElement($items, $delta, $base_element, $form, $form_state);
    }

    return $elements;
  }

  /**
   * { @inheritdoc }
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $item = $items[$delta];
    $commerce_store = $item->commerce_store;

    $store_storage = \Drupal::entityTypeManager()->getStorage('commerce_store');
    $store = $store_storage->load($commerce_store);

    // Determine the target entity type from field settings.
    $target_type = $this->getFieldSetting('target_type');
    $handler_settings = $this->fieldDefinition->getSetting('handler_settings');

    $element += [
      '#type' => 'container',
    ];
    $element['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $store->label(),
      '#default_value' => !empty($item->enabled) ? $item->enabled : 0,
    ];
    $element['commerce_store'] = [
      '#type' => 'value',
      '#value' => $commerce_store,
    ];

    if (empty($handler_settings['target_bundles'])) {
      return $element;
    }

    $bundle = reset($handler_settings['target_bundles']);

    // Retrieve or create the referred entity.
    $entity = $item->entity ?? \Drupal::entityTypeManager()
      ->getStorage($target_type)
      ->create(['type' => $bundle]);

    $checkbox_name = $this->buildEnabledElementName($form, $delta);
    $element['entity_form'] = [
      '#type' => 'inline_entity_form',
      '#entity_type' => $target_type,
      '#bundle' => $bundle,
      '#default_value' => $entity,
      '#form_mode' => 'default',
      '#disable_actions' => TRUE,
      '#states' => [
        'visible' => [
          ":input[name='$checkbox_name']" => ['checked' => TRUE],
        ],
      ],
    ];

    return $element;
  }

  /**
   * Builds an HTML name attribute of the "enabled" checkbox.
   *
   * @param array $form
   *   The full form structure.
   * @param int $delta
   *   The current delta.
   *
   * @return string
   *   Name attribue value.
   */
  protected function buildEnabledElementName(array $form, $delta): string {
    // Get the parents from the full form.
    $parents = isset($form['#parents']) ? $form['#parents'] : [];

    // Append the field name and delta.
    $parents[] = $this->fieldDefinition->getName();
    $parents[] = $delta;
    $parents[] = 'enabled';

    $first = array_shift($parents);
    return $parents ? $first . '[' . implode('][', $parents) . ']' : $first;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state): array {
    $massaged_values = [];
    foreach ($values as $delta => $value) {
      $parents = $form['#parents'];
      $parents[] = $this->fieldDefinition->getName();
      $parents[] = 'widget';
      $parents[] = $delta;

      $element = NestedArray::getValue($form, $parents);

      $massaged_values[$delta] = [
        'enabled' => $value['enabled'],
        'commerce_store' => $value['commerce_store'],
        'entity' => $element['entity_form']['#entity'],
      ];
    }
    return $massaged_values;
  }

}
