<?php

namespace Drupal\commerce_field_per_store\Plugin\SingleContentSyncFieldProcessor;

use Drupal\Core\Entity\FieldableEntityInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\single_content_sync\Plugin\SingleContentSyncFieldProcessor\EntityReferenceRevisions;

/**
 * Plugin for the entity reference per commerce store field processor plugin.
 *
 * @SingleContentSyncFieldProcessor(
 *   id = "commerce_field_per_store",
 *   label = @Translation("Entity reference per commerce store field
 *   processor"), field_type = "commerce_entity_reference_per_store",
 * )
 */
class CommerceEntityReferencePerStore extends EntityReferenceRevisions {

  /**
   * {@inheritdoc}
   */
  public function exportFieldValue(FieldItemListInterface $field): array {
    $paragraph_storage = $this->entityTypeManager->getStorage('paragraph');

    $value = $field->getValue();
    foreach ($value as &$item) {
      if (!empty($item['target_id'])) {
        /** @var \Drupal\paragraphs\Entity\Paragraph $paragraph */
        if ($paragraph = $paragraph_storage->load($item['target_id'])) {
          $item = [
              'enabled' => $item['enabled'],
              'commerce_store' => $item['commerce_store'],
            ] + $this->exporter->doExportToArray($paragraph);
        }
      }
    }

    return $value;
  }


  /**
   * {@inheritdoc}
   */
  public function importFieldValue(FieldableEntityInterface $entity, string $fieldName, array $value): void {
    $values = [];

    foreach ($value as $childEntity) {
      $values[] = [
        'enabled' => $childEntity['enabled'],
        'commerce_store' => $childEntity['commerce_store'],
      ] + $this->buildValue($childEntity);
    }

    $entity->set($fieldName, $values);
  }

  /**
   * Builds the value for a field item.
   */
  protected function buildValue(array $childEntity): array {

    // Import config relation just by setting target id.
    if (isset($childEntity['type']) && $childEntity['type'] === 'config') {
      return [
        'target_id' => $childEntity['value'],
      ];
    }

    // Do not overwrite commerce products.
    if ($childEntity['entity_type'] === 'commerce_product') {
      $referencedEntity = $this
        ->entityRepository
        ->loadEntityByUuid($childEntity['entity_type'], $childEntity['uuid']);
      if ($referencedEntity) {
        return [
          'entity' => $referencedEntity,
        ];
      }
    }

    // If the entity was fully exported we do the full import.
    if ($this->importer->isFullEntity($childEntity)) {
      if ($entity = $this->importer->doImport($childEntity)) {
        return ['entity' => $entity];
      }
    }

    $referencedEntity = $this
      ->entityRepository
      ->loadEntityByUuid($childEntity['entity_type'], $childEntity['uuid']);

    // Create a stub entity without custom field values.
    if (!$referencedEntity) {
      $referencedEntity = $this->importer->createStubEntity($childEntity);
    }

    return ['entity' => $referencedEntity];
  }

}
