<?php

namespace Drupal\commerce_fincra\Service;

use GuzzleHttp\ClientInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\commerce_fincra\Plugin\Commerce\PaymentGateway\FincraStandardInterface;

class FincraCheckoutRedirectService {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * The logger factory.
   *
   * @var \Drupal\Core\Logger\LoggerChannelFactoryInterface
   */
  protected $logger;

  /**
   * Constructs a new FincraCheckoutRedirectService object.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   */
  public function __construct(ClientInterface $http_client, LoggerChannelFactoryInterface $logger) {
    $this->httpClient = $http_client;
    $this->logger = $logger->get('commerce_fincra');
  }

  /**
   * {@inheritdoc}
   */
  public function getBaseUrl(string $mode): string {
    return $mode === 'live'
      ? 'https://api.fincra.com'
      : 'https://sandboxapi.fincra.com';
  }

  /**
   * {@inheritdoc}
   */
  public function createCheckoutRedirect($params, FincraStandardInterface $gateway) {
    $baseUrl = $this->getBaseUrl($gateway->getMode());

    try {
      $response = $this->httpClient->request('POST', $baseUrl . '/checkout/payments', [
        'headers' => [
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
          'api-key' => $gateway->getSecretKey(),
          'x-pub-key' => $gateway->getPublicKey(),
        ],
        'json' => $params,
      ]);

      $body = json_decode($response->getBody()->getContents(), TRUE);
      return $body;
    }
    catch (\Exception $e) {
      $this->logger->error('Fincra checkout error: @message', ['@message' => $e->getMessage()]);
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function verifyTransaction($reference, FincraStandardInterface $gateway) {
    if ($reference === NULL) {
      throw new \InvalidArgumentException('Missing the required parameter $reference when calling verifyTransaction');
    }

    $baseUrl = $this->getBaseUrl($gateway->getMode());

    try {
      $response = $this->httpClient->request('GET', $baseUrl . '/checkout/payments/merchant-reference/' . $reference, [
        'headers' => [
          'Accept' => 'application/json',
          'Content-Type' => 'application/json',
          'api-key' => $gateway->getSecretKey(),
          'x-business-id' => $gateway->getBusinessId(),
        ],
      ]);

      $data = json_decode($response->getBody()->getContents(), TRUE);
      return $data;
    }
    catch (\Exception $e) {
      $this->logger->error('An error occurred while verifying transaction: ' . $reference . '. Error: ' . $e->getMessage());
      return [
        'status' => FALSE,
        'data' => $e->getMessage(),
      ];
    }
  }

}
