<?php

namespace Drupal\commerce_fincra\Controller;

use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\JsonResponse;

/**
 * Provides webhook controller.
 */
class FincraWebhookController extends ControllerBase {

  /**
   * Provides webhook endpoint for Fincra.
   */
  public function handle(Request $request) {
    $payload = $request->getContent();
    $signature = $request->headers->get('signature');

    if (empty($payload) || empty($signature)) {
      return new JsonResponse(['error' => 'Invalid request'], 400);
    }

    // Get the 'fincra_standard' payment gateway.
    $gateway_storage = $this->entityTypeManager()->getStorage('commerce_payment_gateway');

    /** @var \Drupal\commerce_payment\Entity\PaymentGatewayInterface[] $gateways */
    $gateways = $gateway_storage->loadByProperties([
      'plugin' => 'fincra_standard',
      'status' => TRUE,
    ]);

    if (!$gateways) {
      $this->getLogger('commerce_fincra')->error('Fincra webhook: Payment gateway not found.');
      return new JsonResponse(['error' => 'Payment gateway not found'], 400);
    }

    $gateway = reset($gateways);

    // Get plugin and webhook/encryption key.
    $plugin = $gateway->getPlugin();
    if (!method_exists($plugin, 'getWebhookKey')) {
      $this->getLogger('commerce_fincra')->error('Fincra webhook: getWebhookKey() method not found in gateway plugin.');
      return new JsonResponse(['error' => 'The webhook/encryption key method not found'], 400);
    }

    $secretKey = $plugin->getWebhookKey();
    if (empty($secretKey)) {
      $this->getLogger('commerce_fincra')->error('Fincra webhook: Missing webhook/encryption key in gateway configuration.');
      return new JsonResponse(['error' => 'Missing webhook/encryption key'], 400);
    }

    // Build HMAC.
    $computedSignature = hash_hmac('sha512', $payload, $secretKey);

    if (!hash_equals($computedSignature, $signature)) {
      $this->getLogger('commerce_fincra')->warning('Fincra webhook: Invalid signature.');
      return new JsonResponse(['error' => 'Invalid signature'], 403);
    }

    $content = json_decode($payload, TRUE);

    if (!$content || empty($content['event'])) {
      $this->getLogger('commerce_fincra')->error('Fincra webhook: Invalid data structure.');
      return new JsonResponse(['status' => 'error', 'message' => 'Invalid data structure'], 400);
    }

    $event = $content['event'];
    $data = $content['data'] ?? [];

    switch ($event) {
      case 'payout.successful':
      case 'charge.successful':
        $reference = $data['reference'];
        $status    = strtolower($data['status']);

        $payments = $this->entityTypeManager()
          ->getStorage('commerce_payment')
          ->loadByProperties(['remote_id' => $reference]);

        if ($payment = reset($payments)) {
          $mapping = [
            'success'     => 'completed',
            'failed'      => 'failed',
            'abandoned'   => 'authorization_voided',
            'ongoing'     => 'pending',
            'pending'     => 'pending',
            'processing'  => 'pending',
            'queued'      => 'pending',
            'reversed'    => 'refunded',
          ];

          if (isset($mapping[$status])) {
            $payment->setState($mapping[$status]);
            $payment->save();
          }
        }
        $this->getLogger('commerce_fincra')->notice('Fincra webhook: Transaction successful for reference @ref', [
          '@ref' => $data['reference']
        ]);
        break;

      case 'payout.failed':
      case 'charge.failed':
        $reason = $data['reason'] ?? 'Unknown';
        $this->getLogger('commerce_fincra')->warning('Fincra webhook: Transaction failed for reference @ref. Reason: @reason', [
          '@ref' => $data['reference'],
          '@reason' => $reason,
        ]);
        break;

      default:
        $this->getLogger('commerce_fincra')->debug('Fincra webhook: Unhandled Fincra event: @event', ['@event' => $event]);
    }

    return new JsonResponse(['status' => 'ok']);
  }
}
