<?php

namespace Drupal\commerce_fincra\Plugin\Commerce\PaymentGateway;

use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\Core\Form\FormStateInterface;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Fincra Off-site payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "fincra_standard",
 *   label = "Fincra (Off-site)",
 *   display_label = "Fincra",
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_fincra\PluginForm\FincraStandardForm",
 *   },
 *   payment_method_types = {"credit_card"},
 *   credit_card_types = {
 *     "mastercard", "visa",
 *   },
 *   requires_billing_information = TRUE,
 * )
 */
class FincraStandard extends OffsitePaymentGatewayBase implements FincraStandardInterface {

  /**
   * {@inheritdoc}
   */
  public function getSecretKey() {
    $mode = $this->getMode() ?? 'test';
    return $this->configuration["secret_key_{$mode}"] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getPublicKey() {
    $mode = $this->getMode() ?? 'test';
    return $this->configuration["public_key_{$mode}"] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getBusinessId() {
    return $this->configuration['business_id'];
  }

  /**
   * {@inheritdoc}
   */
  public function getWebhookKey() {
    return $this->configuration['webhook_key'];
  }

  /**
   * {@inheritdoc}
   */
  public function debugPayload() {
    return (bool) $this->configuration['debug_payload'];
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
        'secret_key_test' => '',
        'secret_key_live' => '',
        'public_key_test' => '',
        'public_key_live' => '',
        'business_id'     => '',
        'webhook_key'     => '',
        'debug_payload'   => '',
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['secret_key_test'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test Secret Key'),
      '#default_value' => $this->configuration['secret_key_test'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="configuration[fincra_standard][mode]"]' => ['value' => 'test'],
        ],
      ],
    ];
    $form['public_key_test'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test Public Key'),
      '#default_value' => $this->configuration['public_key_test'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="configuration[fincra_standard][mode]"]' => ['value' => 'test'],
        ],
      ],
    ];

    // Live keys.
    $form['secret_key_live'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live Secret Key'),
      '#default_value' => $this->configuration['secret_key_live'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="configuration[fincra_standard][mode]"]' => ['value' => 'live'],
        ],
      ],
    ];
    $form['public_key_live'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Live Public Key'),
      '#default_value' => $this->configuration['public_key_live'] ?? '',
      '#states' => [
        'visible' => [
          ':input[name="configuration[fincra_standard][mode]"]' => ['value' => 'live'],
        ],
      ],
    ];

    // Business ID.
    $form['business_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Business ID'),
      "#description" => $this->t('The unique identifier for a business within the Fincra platform. Your Business ID can be found in the "My Profile" section of your Fincra dashboard.'),
      '#default_value' => $this->getBusinessId(),
      '#required' => TRUE,
    ];

    // Webhook/Encryption Key.
    $form['webhook_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Webhook/Encryption Key'),
      '#default_value' => $this->getWebhookKey(),
      '#required' => TRUE,
    ];

    $webhook_url = Url::fromRoute('commerce_fincra.webhook', [], ['absolute' => TRUE])->toString();
    $form['webhook_info'] = [
      '#type' => 'item',
      '#title' => $this->t('Webhook URL'),
      '#markup' => $this->t('Use this URL in your Fincra dashboard: <code>@url</code>', [
        '@url' => $webhook_url,
      ]),
    ];

    $test_cards_link = Link::fromTextAndUrl(
      $this->t('View test cards'),
      Url::fromUri('https://docs.fincra.com/docs/test-cards', [
        'attributes' => ['target' => '_blank'],
      ])
    )->toString();

    $form['test_mode_note'] = [
      '#type' => 'item',
      '#title' => $this->t('Note'),
      '#markup' => $this->t('Test mode supports only USD and NGN currencies. @test_cards_link.', [
        '@test_cards_link' => $test_cards_link,
      ]),
    ];

    $form['debug_payload'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Debug payload'),
      '#default_value' => $this->configuration['debug_payload'],
      '#description' => $this->t('Enable this to log full API payloads to Drupal logs for debugging purposes. Do not enable on production unless necessary.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $mode = $values['mode'];

      // Get active keys.
      $secret_key = $values["secret_key_{$mode}"];
      $public_key = $values["public_key_{$mode}"];

      // Validate secret key.
      if (!is_string($secret_key)) {
        $form_state->setError($form["secret_key_{$mode}"], $this->t('A valid Fincra secret key must be a string.'));
      }
      elseif (strlen($secret_key) <= 14) {
        $form_state->setError($form["secret_key_{$mode}"], $this->t('The secret key must be longer than 14 characters.'));
      }

      // Validate public key.
      if (!is_string($public_key) || !(substr($public_key, 0, 3) === 'pk_')) {
        $form_state->setError($form["public_key_{$mode}"], $this->t('A valid Fincra public key must start with \'pk_\'.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['secret_key_test'] = $values['secret_key_test'];
      $this->configuration['public_key_test'] = $values['public_key_test'];
      $this->configuration['secret_key_live'] = $values['secret_key_live'];
      $this->configuration['public_key_live'] = $values['public_key_live'];
      $this->configuration['webhook_key']     = $values['webhook_key'];
      $this->configuration['business_id']     = $values['business_id'];
      $this->configuration['debug_payload']   = $values['debug_payload'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    $order_id = $order->id();
    $reference = $request->query->get('reference');
    if (empty($reference)) {
      $this->messenger()->addError($this->t('Could not complete your payment with Fincra. Please try again or contact us if the problem persists.'));
      throw new PaymentGatewayException("Could not find Transaction Reference in return request. Order ID - $order_id.");
    }

    $payment_gateway = $this->parentEntity;
    /** @var \Drupal\commerce_fincra\Plugin\Commerce\PaymentGateway\FincraStandardInterface $plugin */
    $plugin = $payment_gateway->getPlugin();

    /** @var \Drupal\commerce_fincra\Service\FincraCheckoutRedirectService $fincra */
    $fincra = \Drupal::service('commerce_fincra.checkout_redirect');
    $verify_transaction = $fincra->verifyTransaction($reference, $plugin);

    // Checks if the response contains an error.
    if (!$verify_transaction['status']) {
      $this->messenger()->addError($this->t('Could not complete your payment with Fincra. Please try again or contact us if the problem persists.'));
      throw new PaymentGatewayException("Payment was not successful for order ID - $order_id.");
    }
    $remote_state = $verify_transaction['data']['status'];

    // Set the Payer ID used to finalize payment.
    $order_data = $order->getData('fincra_standard');
    $order_data['payer_id'] = $verify_transaction['data']['customer']['name'];
    $order->setData('fincra_standard', $order_data);

    // Set commerce payment data.
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');
    $payment = $payment_storage->create([
      'state' => 'authorization',
      'amount' => $order->getTotalPrice(),
      'payment_gateway' => $payment_gateway->id(),
      'order_id' => $order_id,
      'remote_id' => $verify_transaction['data']['id'],
      'remote_state' => $remote_state,
    ]);

    // Update payment status.
    $status_mapping = $this->getStatusMapping();
    if (isset($status_mapping[$remote_state])) {
      $payment->setState($status_mapping[$remote_state]);
    }
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function onCancel(OrderInterface $order, Request $request) {
    parent::onCancel($order, $request);
    $this->messenger()->addMessage($this->t('Payment canceled.'));
  }

  /**
   * {@inheritdoc}
   */
  public function getStatusMapping($status = NULL) {
    $mapping = [
      'success' => 'completed',
      'abandoned' => 'authorization_voided',
      'failed' => 'refunded',
    ];
    // If a status was passed, return its corresponding payment state.
    if (isset($status) && isset($mapping[$status])) {
      return $mapping[$status];
    }
    return $mapping;
  }
  
}
