<?php

namespace Drupal\commerce_fincra\PluginForm;

use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\PluginForm\PaymentOffsiteForm as BasePaymentOffsiteForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Provides the base class for payment off-site forms.
 */
class FincraStandardForm extends BasePaymentOffsiteForm {
  
  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    
    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->entity;

    /** @var \Drupal\commerce_fincra\Plugin\Commerce\PaymentGateway\FincraStandardInterface $plugin */
    $plugin = $payment->getPaymentGateway()->getPlugin();

    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $payment->getOrder();

    $customer_name = 'Fincra Customer';
    // Adds information about the billing profile.
    if ($billing_profile = $order->getBillingProfile()) {
      /** @var \Drupal\address\AddressInterface $address */
      $address = $billing_profile->get('address')->first();
      $customer_name = $address->getGivenName() . ' ' . $address->getFamilyName();
    }
    
    // Get total order price.
    $amount = $payment->getAmount();

    $params = [
      'reference' => $order->uuid() . '-' . time(),

      'amount' => (float) $amount->getNumber(),
      'currency' => $amount->getCurrencyCode(),

      'customer' => [
        'name' => $customer_name,
        'email' => $order->getEmail(),
      ],
      'feeBearer' => 'business',

      'redirectUrl' => $form['#return_url'],
      'cancelUrl' => $form['#cancel_url'],

      'paymentMethods' => ['card', 'bank_transfer'],
      'defaultPaymentMethod' => 'card',
    ];

    // Only card payment method is available in TEST mode for USD.
    if ($plugin->getMode() == 'test' && $amount->getCurrencyCode() == 'USD') {
      $params['paymentMethods'] = ['card'];
    }

    // Only bank_transfer payment method is available in TEST mode for GHS.
    if ($plugin->getMode() == 'test' && $amount->getCurrencyCode() == 'GHS') {
      $params['paymentMethods'] = ['bank_transfer'];
      $params['defaultPaymentMethod'] = 'bank_transfer';
    }

    if (isset($fields)) {
      $params['metadata']['custom_fields'] = $fields;
    }

    if ($plugin->debugPayload()) {
      $json_payload = json_encode($params);
      \Drupal::logger('commerce_fincra')->notice('Fincra Payload: @payload', [
        '@payload' => $json_payload,
      ]);
    }

    /** @var \Drupal\commerce_fincra\Service\FincraCheckoutRedirectService $fincra */
    $fincra = \Drupal::service('commerce_fincra.checkout_redirect');

    // Initialize a transaction.
    $response = $fincra->createCheckoutRedirect($params, $plugin);

    if (!$response) {
      throw new PaymentGatewayException('Unable to initialize payment with Fincra. Please try again or contact customer support.');
    }

    $redirect_url = $response['data']['link'];
    $order->setData('fincra_standard', [
      'reference' => $response['data']['reference'],
      'access_code' => $response['data']['payCode'],
      'authorization_url' => $redirect_url,
    ]);
    $order->save();
    
    $data = [
      'return' => $form['#return_url'],
      'cancel' => $form['#cancel_url'],
      'total' => $payment->getAmount()->getNumber(),
    ];

    $redirect_method = BasePaymentOffsiteForm::REDIRECT_GET;

    return $this->buildRedirectForm($form, $form_state, $redirect_url, $data, $redirect_method);
  }
  
}
