<?php

namespace Drupal\commerce_hyperpay\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_hyperpay\Event\AlterHyperpayAmountEvent;
use Drupal\commerce_hyperpay\Event\HyperpayPaymentEvents;
use Drupal\commerce_hyperpay\Transaction\Status\Factory;
use Drupal\commerce_hyperpay\Transaction\Status\Rejected;
use Drupal\commerce_hyperpay\Transaction\Status\SuccessOrPending;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\InvalidRequestException;
use Drupal\commerce_payment\Exception\InvalidResponseException;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Render\RenderContext;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\RequestOptions;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Hyperpay applepay payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "hyperpay_applepay_payment",
 *   label = @Translation("HyperPay ApplePay Payment"),
 *   display_label = @Translation("HyperPay ApplePay Payment"),
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_hyperpay\PluginForm\HyperPayCopyAndPayForm",
 *   },
 * )
 */
class HyperPayApplePay extends OffsitePaymentGatewayBase implements HyperPayInterface {

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * The http client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The payment storage.
   *
   * @var \Drupal\commerce_payment\PaymentStorageInterface
   */
  protected $paymentStorage;

  /**
   * The price rounder.
   *
   * @var \Drupal\commerce_price\RounderInterface
   */
  protected $rounder;

  /**
   * The renderer.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->eventDispatcher = $container->get('event_dispatcher');
    $instance->httpClient = $container->get('http_client');
    $instance->paymentStorage = $instance->entityTypeManager->getStorage('commerce_payment');
    $instance->rounder = $container->get('commerce_price.rounder');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'server' => 'default',
      'allowed_cards' => [],
      'auth_bearer' => '',
      'entity_id' => '',
      'widget_style' => 'card',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['server'] = [
      '#type' => 'select',
      '#title' => $this->t('Server'),
      '#default_value' => $this->configuration['server'],
      '#options' => [
        'default' => 'Default (oppwa.com)',
        'eu' => 'Europe (eu-prod.oppwa.com)',
      ],
    ];
    $form['allowed_cards'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed Cards'),
      '#options' => [
        'APPLEPAY' => 'APPLEPAY',
      ],
      '#default_value' => $this->configuration['allowed_cards'],
      '#required' => TRUE,
    ];
    $form['auth_bearer'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Authorization Bearer'),
      '#default_value' => $this->configuration['auth_bearer'],
      '#required' => TRUE,
    ];
    $form['entity_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Entity ID'),
      '#default_value' => $this->configuration['entity_id'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['auth_bearer'] = $values['auth_bearer'];
      $this->configuration['entity_id'] = $values['entity_id'];
      $this->configuration['mode'] = $values['mode'];
      $this->configuration['server'] = $values['server'];
      $this->configuration['allowed_cards'] = $values['allowed_cards'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCheckout(array $params = []) {
    $checkout_data = NULL;

    $json_response = $this->sendRequest('/v1/checkouts', 'POST', $params);
    if (!empty($json_response['id'])) {
      $checkout_data['checkout_id'] = $json_response['id'];
      $checkout_data['integrity'] = $json_response['integrity'];
    }
    else {
      throw new InvalidRequestException($this->t('Cannot prepare Hyperpay checkout: could not retrieve checkout ID.'));
    }

    return $checkout_data;
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    $this->assertPaymentState($payment, ['new']);
    $payment_method = $payment->getPaymentMethod();
    $this->assertPaymentMethod($payment_method);

    // Perform the create payment request here, throw an exception if it fails.
    // See \Drupal\commerce_payment\Exception for the available exceptions.
    // Remember to take into account $capture when performing the request.
    $payment_method_token = $payment_method->getRemoteId();
    $order = $payment->getOrder();
    $payment_amount = $this->getPayableAmount($order);
    $params = [
      'entityId' => $this->configuration['entity_id_recurring'],
      'amount' => $payment_amount->getNumber(),
      'currency' => $payment_amount->getCurrencyCode(),
      'paymentType' => 'DB',
    ];
    $json_response = $this->sendRequest('/v1/registrations/' . $payment_method_token . '/payments', 'POST', $params);
    if (empty($json_response['id'])) {
      throw new InvalidResponseException($this->t('Unable to identify Hyperpay payment (resource path: @resource_path)', ['@resource_path' => $resource_path]));
    }

    $payment->setRemoteId($json_response['id']);

    if (!isset($json_response['amount']) && !isset($json_response['currency'])) {
      throw new InvalidResponseException($json_response['result']['description']);
    }

    $paid_amount = new Price($json_response['amount'], $json_response['currency']);
    if (!$paid_amount->equals($payment->getAmount())) {
      throw new InvalidResponseException($this->t('The payment amount deviates from the expected value (given: @given_currency @given_amount / expected: @expected_currency @expected_amount).',
        [
          '@given_currency'    => $paid_amount->getCurrencyCode(),
          '@given_amount'      => $paid_amount->getNumber(),
          '@expected_currency' => $payment->getAmount()->getCurrencyCode(),
          '@expected_amount'   => $payment->getAmount()->getNumber(),
        ]
      ));
    }

    $payment_status = Factory::newInstance($json_response['result']['code'], $json_response['result']['description']);
    if (empty($payment_status)) {
      throw new PaymentGatewayException($this->t('Received unknown payment status @code for checkout ID @remote_id (@description).',
        [
          '@code' => $json_response['result']['code'],
          '@remote_id' => $remote_payment_id,
          '@description' => $json_response['result']['description'],
        ]
      ));
    }

    $payment->setRemoteState($payment_status->getCode());
    if ($payment_status instanceof SuccessOrPending) {
      $authorize_transition = $payment->getState()->getWorkflow()->getTransition('authorize_capture');
      $payment->getState()->applyTransition($authorize_transition);
    }

    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    parent::onReturn($order, $request);

    $resource_path = $request->query->get('resourcePath');
    if (empty($resource_path)) {
      throw new PaymentGatewayException('No resource path found in query string on Hyperpay payment return.');
    }
    $checkout_id = $request->query->get('id');
    if (empty($checkout_id)) {
      throw new PaymentGatewayException('No checkout ID specified in query string on Hyperpay payment return.');
    }

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->paymentStorage->loadByRemoteId($checkout_id);
    if (empty($payment)) {
      throw new PaymentGatewayException($this->t('No pre-authorized payment could be found for the checkout ID specified by Hyperpay payment return callback (ID: @checkout_id / resource path: @resource_path)',
        [
          '@checkout_id' => $checkout_id,
          '@resource_path' => $resource_path,
        ])
      );
    }

    $json_response = $this->sendRequest($resource_path, 'GET', []);
    if (empty($json_response['id'])) {
      throw new InvalidResponseException($this->t('Unable to identify Hyperpay payment (resource path: @resource_path)', ['@resource_path' => $resource_path]));
    }

    // The ID we receive in this response is different to the checkout ID.
    // The checkout ID was only a temporary remote value, in order to be able
    // to fetch the payment in this callback. Now, we have received the real
    // remote ID and will use it.
    $remote_payment_id = $json_response['id'];
    $payment->setRemoteId($remote_payment_id);
    if ((int) $payment->getOrderId() !== (int) $order->id()) {
      throw new InvalidResponseException($this->t('The order ID used on the payment return callback (@request_order_id) does not match the parent order ID of the given payment (@payment_order_id). (resource path: @resource_path)',
        [
          '@request_order_id' => $order->id(),
          '@payment_order_id' => $payment->getOrderId(),
          '@resource_path' => $resource_path,
        ]
      ));
    }

    if (!isset($json_response['amount']) && !isset($json_response['currency'])) {
      throw new InvalidResponseException($json_response['result']['description']);
    }

    $paid_amount = new Price($json_response['amount'], $json_response['currency']);
    if (!$paid_amount->equals($payment->getAmount())) {
      throw new InvalidResponseException($this->t('The payment amount deviates from the expected value (given: @given_currency @given_amount / expected: @expected_currency @expected_amount).',
        [
          '@given_currency'    => $paid_amount->getCurrencyCode(),
          '@given_amount'      => $paid_amount->getNumber(),
          '@expected_currency' => $payment->getAmount()->getCurrencyCode(),
          '@expected_amount'   => $payment->getAmount()->getNumber(),
        ]
      ));
    }

    $payment_status = Factory::newInstance($json_response['result']['code'], $json_response['result']['description']);
    if (empty($payment_status)) {
      throw new PaymentGatewayException($this->t('Received unknown payment status @code for checkout ID @remote_id (@description).',
        [
          '@code' => $json_response['result']['code'],
          '@remote_id' => $remote_payment_id,
          '@description' => $json_response['result']['description'],
        ]
      ));
    }

    $payment->setRemoteState($payment_status->getCode());
    if ($payment_status instanceof SuccessOrPending) {
      $capture_transition = $payment->getState()->getWorkflow()->getTransition('capture');
      $payment->getState()->applyTransition($capture_transition);
    }
    elseif ($payment_status instanceof Rejected) {
      throw new PaymentGatewayException('Payment failed at the payment server. Please review your information and try again.');
    }
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, Price $amount = NULL) {
    // @todo Implement refundPayment() method.
  }

  /**
   * {@inheritdoc}
   */
  public function getApiUrl() {
    switch ($this->configuration['server']) {
      case 'default':
        return $this->getMode() == 'test' ? 'https://test.oppwa.com' : 'https://oppwa.com';

      break;
      case 'eu':
        return $this->getMode() == 'test' ? 'https://eu-test.oppwa.com' : 'https://eu-prod.oppwa.com';

      break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function calculateCheckoutIdExpireTime($request_time = NULL) {
    if (empty($request_time)) {
      $request_time = $this->time->getRequestTime();
    }
    // A checkout ID is valid for 30 minutes.
    // @see https://hyperpay.docs.oppwa.com/support/widget
    return $request_time + (30 * 60);
  }

  /**
   * {@inheritdoc}
   */
  public function getPayableAmount(OrderInterface $order) {
    $event = new AlterHyperpayAmountEvent($order);
    $this->eventDispatcher->dispatch($event, HyperpayPaymentEvents::ALTER_AMOUNT);
    $payment_amount = $event->getPaymentAmount();

    return $this->rounder->round($payment_amount);
  }

  /**
   * {@inheritdoc}
   */
  protected function sendRequest($resource_path, $method = 'POST', $params = []) {
    $base_url = $this->getApiUrl();
    $url = $base_url . $resource_path;
    if (!isset($params['entityId'])) {
      $params += [
        'entityId' => $this->configuration['entity_id'],
      ];
    }
    $headers = [
      'Authorization' => "Bearer " . $this->configuration['auth_bearer'],
    ];
    try {
      if ($method == 'POST') {
        $response = $this->httpClient->post($url, [RequestOptions::FORM_PARAMS => $params, RequestOptions::HEADERS => $headers]);
      }
      else {
        $response = $this->httpClient->get($url, [RequestOptions::QUERY => $params, RequestOptions::HEADERS => $headers]);
      }
      return json_decode($response->getBody(), TRUE);
    }
    catch (RequestException $request_exception) {
      throw new InvalidResponseException($this->t('Error occurred on calling the specified Hyperpay resource path @resource_path with message: @msg', ['@resource_path' => $resource_path, '@msg' => $request_exception->getMessage()]));
    }
    catch (\Exception $ex) {
      throw new InvalidResponseException($this->t('Error occurred on calling the specified Hyperpay resource path @resource_path with message: @msg', ['@resource_path' => $resource_path, '@msg' => $ex->getMessage()]));
    }

  }

  /**
   * {@inheritdoc}
   */
  public function checkPaymentStatus($order_id, array $params = []) {
    $mode = $this->getMode();

    if (!isset($params['entityId'])) {
      $params += [
        'entityId' => $this->configuration['entity_id'],
      ];
    }

    $commerce_order = $this->entityTypeManager->getStorage('commerce_order')->load($order_id);
    $checkout_id = $commerce_order->field_hy_checkout_id->value;
    $json_response = $this->sendRequest("/v1/checkouts/{$checkout_id}/payment", 'GET', $params);
    if (empty($json_response['id'])) {
      throw new InvalidResponseException($this->t('Unable to identify Hyperpay payment (resource path: @payment_id)', ['@payment_id' => $payment_id]));
    }
    // Get the payment from the order.
    $payment = $this->entityTypeManager->getStorage('commerce_payment')->loadByProperties(["order_id" => $order_id]);
    $payment = reset($payment);

    $result_code = $json_response['result']['code'];

    if (($mode == 'test' && $result_code == "000.100.110") || ($mode == 'live' && $result_code == "000.000.000")) {
      $authorize_transition = $payment->getState()->getWorkflow()->getTransition('authorize');
      $payment->getState()->applyTransition($authorize_transition);
      $capture_transition = $payment->getState()->getWorkflow()->getTransition('capture');
      $payment->getState()->applyTransition($capture_transition);
      $payment->save();

      $commerce_order->getState()->applyTransitionById('place');
      $context = new RenderContext();
      /** @var \Drupal\Core\Cache\CacheableDependencyInterface $result */
      $result = $this->renderer->executeInRenderContext($context, function () use ($commerce_order) {
        // This code that we don't don't control that in turn triggers early rendering.
        return $commerce_order->save();
      });
      // Handle any bubbled cacheability metadata.
      if (!$context->isEmpty()) {
        $bubbleable_metadata = $context->pop();
        BubbleableMetadata::createFromObject($result)
          ->merge($bubbleable_metadata);
      }
    }
    return $json_response;
  }

}
