<?php

namespace Drupal\commerce_hyperpay\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_hyperpay\Event\AlterHyperpayAmountEvent;
use Drupal\commerce_hyperpay\Event\HyperpayPaymentEvents;
use Drupal\commerce_hyperpay\Transaction\Status\Factory;
use Drupal\commerce_hyperpay\Transaction\Status\Rejected;
use Drupal\commerce_hyperpay\Transaction\Status\SuccessOrPending;
use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\CreditCard;
use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Entity\PaymentMethodInterface;
use Drupal\commerce_payment\Exception\InvalidRequestException;
use Drupal\commerce_payment\Exception\InvalidResponseException;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsStoredPaymentMethodsInterface;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;
use GuzzleHttp\Exception\RequestException;
use GuzzleHttp\RequestOptions;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Hyperpay CopyAndPay payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "hyperpay_copyandpay_payment",
 *   label = @Translation("Hyperpay COPYandPAY Payment"),
 *   display_label = @Translation("Hyperpay COPYandPAY Payment"),
 *   forms = {
 *     "offsite-payment" = "Drupal\commerce_hyperpay\PluginForm\HyperPayCopyAndPayForm",
 *   },
 *   payment_method_types = {"credit_card"},
 *   credit_card_types = {
 *     "amex", "dinersclub", "discover", "jcb", "maestro", "mastercard", "visa",
 *   },
 * )
 */
class HyperPayCopyAndPay extends OffsitePaymentGatewayBase implements HyperPayInterface, SupportsStoredPaymentMethodsInterface {

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * The http client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The payment storage.
   *
   * @var \Drupal\commerce_payment\PaymentStorageInterface
   */
  protected $paymentStorage;

  /**
   * The price rounder.
   *
   * @var \Drupal\commerce_price\RounderInterface
   */
  protected $rounder;

  /**
   * The The uuid service.
   *
   * @var \Drupal\Component\Uuid\UuidInterface
   */
  protected $uuidService;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);

    $instance->eventDispatcher = $container->get('event_dispatcher');
    $instance->httpClient = $container->get('http_client');
    $instance->paymentStorage = $instance->entityTypeManager->getStorage('commerce_payment');
    $instance->rounder = $container->get('commerce_price.rounder');
    $instance->uuidService = $container->get('uuid');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'test_mode_type' => 'EXTERNAL',
      'server' => 'default',
      'allowed_cards' => [],
      'auth_bearer' => '',
      'entity_id' => '',
      'reuse_payment_method' => FALSE,
      'recurring_options' => 'MOTO',
      'entity_id_recurring' => '',
      'widget_style' => 'Card',
    ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);
    $form['test_mode_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Test Mode Type'),
      '#default_value' => $this->configuration['test_mode_type'],
      '#options' => [
        'EXTERNAL' => 'EXTERNAL',
        'INTERNAL' => 'INTERNAL',
        'NONE' => 'NONE',
      ],
      '#states' => [
        'visible' => [
          ":input[name='configuration[hyperpay_copyandpay_payment][mode]']" => ['value' => 'test'],
        ],
        'required' => [
          ":input[name='configuration[hyperpay_copyandpay_payment][mode]']" => ['value' => 'test'],
        ],
      ],
    ];
    $form['server'] = [
      '#type' => 'select',
      '#title' => $this->t('Server'),
      '#default_value' => $this->configuration['server'],
      '#options' => [
        'default' => 'Default (oppwa.com)',
        'eu' => 'Europe (eu-prod.oppwa.com)',
      ],
    ];
    $form['allowed_cards'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Allowed Cards'),
      '#options' => [
        'VISA' => 'VISA',
        'MASTER' => 'MASTER',
        'MASTERDEBIT' => 'MASTERDEBIT',
        'MAESTRO' => 'MAESTRO',
        'DISCOVER' => 'DISCOVER',
        'AMEX' => 'AMEX',
        'MADA' => 'MADA',
      ],
      '#default_value' => $this->configuration['allowed_cards'],
      '#required' => TRUE,
    ];
    $form['auth_bearer'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Authorization Bearer'),
      '#default_value' => $this->configuration['auth_bearer'],
      '#required' => TRUE,
    ];
    $form['entity_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Entity ID'),
      '#default_value' => $this->configuration['entity_id'],
      '#required' => TRUE,
    ];
    $form['reuse_payment_method'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow reusing payment method'),
      '#default_value' => $this->configuration['reuse_payment_method'],
      '#required' => FALSE,
    ];
    $form['recurring_options'] = [
      '#type' => 'select',
      '#title' => $this->t('Recurring options'),
      '#default_value' => $this->configuration['recurring_options'],
      '#options' => [
        'MOTO' => 'MOTO',
        'RECURRING' => 'RECURRING',
      ],
      '#states' => [
        'visible' => [
          ":input[name='configuration[hyperpay_copyandpay_payment][reuse_payment_method]']" => ['checked' => TRUE],
        ],
        'required' => [
          ":input[name='configuration[hyperpay_copyandpay_payment][reuse_payment_method]']" => ['checked' => TRUE],
        ],
      ],
    ];
    $form['entity_id_recurring'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Entity ID (Recurring - no 3D Secure)'),
      '#default_value' => $this->configuration['entity_id_recurring'],
      '#states' => [
        'visible' => [
          ":input[name='configuration[hyperpay_copyandpay_payment][reuse_payment_method]']" => ['checked' => TRUE],
        ],
        'required' => [
          ":input[name='configuration[hyperpay_copyandpay_payment][reuse_payment_method]']" => ['checked' => TRUE],
        ],
      ],
    ];
    $form['widget_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Widget Style'),
      '#default_value' => $this->configuration['widget_style'],
      '#options' => [
        'card' => 'Card',
        'plain' => 'Plain',
      ],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['auth_bearer'] = $values['auth_bearer'];
      $this->configuration['allowed_cards'] = $values['allowed_cards'];
      $this->configuration['entity_id'] = $values['entity_id'];
      $this->configuration['reuse_payment_method'] = $values['reuse_payment_method'];
      $this->configuration['entity_id_recurring'] = $values['entity_id_recurring'];
      $this->configuration['recurring_options'] = $values['recurring_options'];
      $this->configuration['mode'] = $values['mode'];
      $this->configuration['server'] = $values['server'];
      $this->configuration['test_mode_type'] = $values['test_mode_type'];
      $this->configuration['widget_style'] = $values['widget_style'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCheckout(array $params = []) {
    $checkout_data = NULL;
    if ($this->configuration['reuse_payment_method']) {
      $params += [
        'createRegistration' => TRUE,
      ];
      if ($this->configuration['recurring_options'] == 'RECURRING') {
        $params += [
          'customParameters[recurringPaymentAgreement]' => $this->uuidService->generate(),
          'standingInstruction.mode' => 'INITIAL',
          'standingInstruction.source' => 'CIT',
          'standingInstruction.type' => 'UNSCHEDULED',
        ];
      }
      else {
        $params += [
          'recurringType' => 'INITIAL',
        ];
      }

    }
    $json_response = $this->sendRequest('/v1/checkouts', 'POST', $params);
    if (!empty($json_response['id'])) {
      $checkout_data['checkout_id'] = $json_response['id'];
      $checkout_data['integrity'] = $json_response['integrity'];
    }
    else {
      throw new InvalidRequestException($this->t('Cannot prepare Hyperpay checkout: could not retrieve checkout ID.'));
    }

    return $checkout_data;
  }

  /**
   * {@inheritdoc}
   */
  public function createPaymentMethod(PaymentMethodInterface $payment_method, Request $request) {
    $payment_method->save();
    return $payment_method;
  }

  /**
   * {@inheritdoc}
   */
  public function deletePaymentMethod(PaymentMethodInterface $payment_method) {
    $payment_method->delete();
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $capture = TRUE) {
    $this->assertPaymentState($payment, ['new']);
    $payment_method = $payment->getPaymentMethod();
    $this->assertPaymentMethod($payment_method);

    // Perform the create payment request here, throw an exception if it fails.
    // See \Drupal\commerce_payment\Exception for the available exceptions.
    // Remember to take into account $capture when performing the request.
    $payment_method_token = $payment_method->getRemoteId();
    $order = $payment->getOrder();
    $payment_amount = $this->getPayableAmount($order);
    $params = [
      'entityId' => $this->configuration['entity_id_recurring'],
      'amount' => $payment_amount->getNumber(),
      'currency' => $payment_amount->getCurrencyCode(),
      'paymentType' => 'DB',
      'recurringType' => 'REPEATED',
    ];
    if ($this->configuration['recurring_options'] == 'RECURRING') {
      $params += [
        'merchantTransactionId' => $order->id(),
        'customParameters[3DS2_enrolled]' => TRUE,
        'customParameters[recurringPaymentAgreement]' => $payment_method->uuid(),
        'standingInstruction.mode' => 'REPEATED',
        'standingInstruction.type' => 'UNSCHEDULED',
        'standingInstruction.source' => 'MIT',
      ];
    }
    $json_response = $this->sendRequest('/v1/registrations/' . $payment_method_token . '/payments', 'POST', $params);
    if (empty($json_response['id'])) {
      throw new InvalidResponseException($this->t('Unable to identify Hyperpay payment (resource path: @resource_path)', ['@resource_path' => $resource_path]));
    }

    $payment->setRemoteId($json_response['id']);

    if (!isset($json_response['amount']) && !isset($json_response['currency'])) {
      throw new InvalidResponseException($json_response['result']['description']);
    }

    $paid_amount = new Price($json_response['amount'], $json_response['currency']);
    if (!$paid_amount->equals($payment->getAmount())) {
      throw new InvalidResponseException($this->t('The payment amount deviates from the expected value (given: @given_currency @given_amount / expected: @expected_currency @expected_amount).',
        [
          '@given_currency'    => $paid_amount->getCurrencyCode(),
          '@given_amount'      => $paid_amount->getNumber(),
          '@expected_currency' => $payment->getAmount()->getCurrencyCode(),
          '@expected_amount'   => $payment->getAmount()->getNumber(),
        ]
      ));
    }

    $payment_status = Factory::newInstance($json_response['result']['code'], $json_response['result']['description']);
    if (empty($payment_status)) {
      throw new PaymentGatewayException($this->t('Received unknown payment status @code for checkout ID @remote_id (@description).',
        [
          '@code' => $json_response['result']['code'],
          '@remote_id' => $remote_payment_id,
          '@description' => $json_response['result']['description'],
        ]
      ));
    }

    $payment->setRemoteState($payment_status->getCode());
    if ($payment_status instanceof SuccessOrPending) {
      $capture_transition = $payment->getState()->getWorkflow()->getTransition('authorize_capture');
      $payment->getState()->applyTransition($capture_transition);
    }

    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {
    parent::onReturn($order, $request);

    $resource_path = $request->query->get('resourcePath');
    if (empty($resource_path)) {
      throw new PaymentGatewayException('No resource path found in query string on Hyperpay payment return.');
    }
    $checkout_id = $request->query->get('id');
    if (empty($checkout_id)) {
      throw new PaymentGatewayException('No checkout ID specified in query string on Hyperpay payment return.');
    }

    /** @var \Drupal\commerce_payment\Entity\PaymentInterface $payment */
    $payment = $this->paymentStorage->loadByRemoteId($checkout_id);
    if (empty($payment)) {
      throw new PaymentGatewayException($this->t('No pre-authorized payment could be found for the checkout ID specified by Hyperpay payment return callback (ID: @checkout_id / resource path: @resource_path)',
        [
          '@checkout_id' => $checkout_id,
          '@resource_path' => $resource_path,
        ])
      );
    }

    $json_response = $this->sendRequest($resource_path, 'GET', []);
    if (empty($json_response['id'])) {
      throw new InvalidResponseException($this->t('Unable to identify Hyperpay payment (resource path: @resource_path)', ['@resource_path' => $resource_path]));
    }

    // The ID we receive in this response is different to the checkout ID.
    // The checkout ID was only a temporary remote value, in order to be able
    // to fetch the payment in this callback. Now, we have received the real
    // remote ID and will use it.
    $remote_payment_id = $json_response['id'];
    $payment->setRemoteId($remote_payment_id);
    if ((int) $payment->getOrderId() !== (int) $order->id()) {
      throw new InvalidResponseException($this->t('The order ID used on the payment return callback (@request_order_id) does not match the parent order ID of the given payment (@payment_order_id). (resource path: @resource_path)',
        [
          '@request_order_id' => $order->id(),
          '@payment_order_id' => $payment->getOrderId(),
          '@resource_path' => $resource_path,
        ]
      ));
    }

    if (!isset($json_response['amount']) && !isset($json_response['currency'])) {
      throw new InvalidResponseException($json_response['result']['description']);
    }

    $paid_amount = new Price($json_response['amount'], $json_response['currency']);
    if (!$paid_amount->equals($payment->getAmount())) {
      throw new InvalidResponseException($this->t('The payment amount deviates from the expected value (given: @given_currency @given_amount / expected: @expected_currency @expected_amount).',
        [
          '@given_currency'    => $paid_amount->getCurrencyCode(),
          '@given_amount'      => $paid_amount->getNumber(),
          '@expected_currency' => $payment->getAmount()->getCurrencyCode(),
          '@expected_amount'   => $payment->getAmount()->getNumber(),
        ]
      ));
    }

    $payment_status = Factory::newInstance($json_response['result']['code'], $json_response['result']['description']);
    if (empty($payment_status)) {
      throw new PaymentGatewayException($this->t('Received unknown payment status @code for checkout ID @remote_id (@description).',
        [
          '@code' => $json_response['result']['code'],
          '@remote_id' => $remote_payment_id,
          '@description' => $json_response['result']['description'],
        ]
      ));
    }

    $payment->setRemoteState($payment_status->getCode());
    if ($payment_status instanceof SuccessOrPending) {
      $capture_transition = $payment->getState()->getWorkflow()->getTransition('capture');
      $payment->getState()->applyTransition($capture_transition);
      // Create payment method if it's allowed.
      if ($this->configuration['reuse_payment_method']) {
        $payment_method_storage = $this->entityTypeManager->getStorage('commerce_payment_method');
        // assert($payment_method_storage instanceof PaymentMethodStorageInterface);.
        $payment_method = $payment_method_storage->createForCustomer(
          'credit_card',
          $this->parentEntity->id(),
          $order->getCustomerId(),
          $order->getBillingProfile()
        );
        if ($this->configuration['recurring_options'] == 'RECURRING') {
          $payment_method->uuid = $json_response["customParameters"]["recurringPaymentAgreement"];
        }
        $payment_method->card_type = strtolower($json_response['paymentBrand']);
        // Check if card type is master, change it to mastercard.
        if ($payment_method->card_type->value == 'master') {
          $payment_method->card_type = 'mastercard';
        }
        $payment_method->card_number = $json_response['card']['last4Digits'];
        $payment_method->card_exp_month = $json_response['card']['expiryMonth'];
        $payment_method->card_exp_year = $json_response['card']['expiryYear'];
        $expires = CreditCard::calculateExpirationTimestamp($payment_method->card_exp_month->value, $payment_method->card_exp_year->value);
        $payment_method->setExpiresTime($expires);
        $payment_method->setRemoteId($json_response['registrationId']);
        $payment_method = $this->createPaymentMethod($payment_method, $request);
        $order->set('payment_method', $payment_method->id());
      }
    }
    elseif ($payment_status instanceof Rejected) {
      throw new PaymentGatewayException('Payment failed at the payment server. Please review your information and try again.');
    }
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, Price $amount = NULL) {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    // If not specified, refund the entire amount.
    $amount = $amount ?: $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    // Perform the refund request here, throw an exception if it fails.
    $remote_id = $payment->getRemoteId();
    $decimal_amount = $amount->getNumber();
    $params = [
      'amount' => $decimal_amount,
      'currency' => $payment->getAmount()->getCurrencyCode(),
      'paymentType' => 'RF',
    ];
    $json_response = $this->sendRequest('/v1/payments/' . $remote_id, 'POST', $params);

    if (!empty($json_response['id'])) {
      $checkout_id = $json_response['id'];
    }
    else {
      throw new InvalidRequestException($this->t('Cannot refund Hyperpay payment: ID does not exist.'));
    }

    // Determine whether payment has been fully or partially refunded.
    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->setState('partially_refunded');
    }
    else {
      $payment->setState('refunded');
    }

    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function getApiUrl() {
    switch ($this->configuration['server']) {
      case 'default':
        return $this->getMode() == 'test' ? 'https://test.oppwa.com' : 'https://oppwa.com';

      break;
      case 'eu':
        return $this->getMode() == 'test' ? 'https://eu-test.oppwa.com' : 'https://eu-prod.oppwa.com';

      break;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function calculateCheckoutIdExpireTime($request_time = NULL) {
    if (empty($request_time)) {
      $request_time = $this->time->getRequestTime();
    }
    // A checkout ID is valid for 30 minutes.
    // @see https://hyperpay.docs.oppwa.com/support/widget
    return $request_time + (30 * 60);
  }

  /**
   * {@inheritdoc}
   */
  public function getPayableAmount(OrderInterface $order) {
    $event = new AlterHyperpayAmountEvent($order);
    $this->eventDispatcher->dispatch($event, HyperpayPaymentEvents::ALTER_AMOUNT);
    $payment_amount = $event->getPaymentAmount();

    return $this->rounder->round($payment_amount);
  }

  /**
   * {@inheritdoc}
   */
  protected function sendRequest($resource_path, $method = 'POST', $params = []) {
    $base_url = $this->getApiUrl();
    $url = $base_url . $resource_path;
    if (!isset($params['entityId'])) {
      $params += [
        'entityId' => $this->configuration['entity_id'],
      ];
    }
    if ($this->getMode() == 'test' && $method == 'POST'&& $this->configuration['test_mode_type'] != 'NONE') {
      $params['testMode'] = $this->configuration['test_mode_type'];
    }
    $headers = [
      'Authorization' => "Bearer " . $this->configuration['auth_bearer'],
    ];
    try {
      if ($method == 'POST') {
        $response = $this->httpClient->post($url, [RequestOptions::FORM_PARAMS => $params, RequestOptions::HEADERS => $headers]);
      }
      else {
        $response = $this->httpClient->get($url, [RequestOptions::QUERY => $params, RequestOptions::HEADERS => $headers]);
      }
      return json_decode($response->getBody(), TRUE);
    }
    catch (RequestException $request_exception) {
      throw new InvalidResponseException($this->t('Error occurred on calling the specified Hyperpay resource path @resource_path with message: @msg', ['@resource_path' => $resource_path, '@msg' => $request_exception->getMessage()]));
    }
    catch (\Exception $ex) {
      throw new InvalidResponseException($this->t('Error occurred on calling the specified Hyperpay resource path @resource_path with message: @msg', ['@resource_path' => $resource_path, '@msg' => $ex->getMessage()]));
    }

  }

}
