<?php

namespace Drupal\commerce_ifthenpay_cc\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_order\Entity\OrderInterface;
use Drupal\commerce_payment\Exception\PaymentGatewayException;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\OffsitePaymentGatewayBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides the Commerce Ifthenpay Credit Card Redirect payment gateway.
 * Docs:
 * https://docs.drupalcommerce.org/commerce2/developer-guide/payments/create-payment-gateway/off-site-gateways/off-site-redirect
 *
 * @CommercePaymentGateway(
 *   id = "ifthenpay_cc",
 *   label = "Ifthenpay - Credit Card",
 *   display_label = "Credit Card",
 *   forms = {
 *     "offsite-payment" =
 *   "Drupal\commerce_ifthenpay_cc\PluginForm\IfThenPayCCForm",
 *   },
 *   payment_method_types = {"credit_card"},
 *   credit_card_types = {
 *     "amex", "maestro", "mastercard", "visa",
 *   },
 * )
 */
class IfthenpayCC extends OffsitePaymentGatewayBase {

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The logger channel.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    /** @var static $gateway */
    $gateway = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $gateway->httpClient = $container->get('http_client');
    $gateway->logger = $container->get('logger.channel.commerce_ifthenpay_cc');
    return $gateway;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
        'cccard_key' => '',
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['cccard_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CCARD Key'),
      '#default_value' => $this->configuration['cccard_key'],
      '#required' => TRUE,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);
    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['cccard_key'] = $values['cccard_key'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function onReturn(OrderInterface $order, Request $request) {

    $status = $request->get('status');
    $remote_id = $request->get('requestId');
    $charged_amount = $request->get('amount');
    $sk = $request->get('sk');
    $order_id = $request->get('id');

    /** @var \Drupal\commerce_payment\PaymentStorage $payment_storage */
    $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');

    /** @var \Drupal\commerce_payment\Entity\Payment $payment */
    $payment = $payment_storage->loadByRemoteId($remote_id);

    if (!$payment) {
      throw new PaymentGatewayException('Payment not found for requestId: ' . $remote_id);
    }

    $order_amount = $order->getTotalPrice()->getNumber();

    // If the errorUrl or cancelUrl (no SK parameter), throw exception
    // and that will abort the order checkout and redirect to previous checkout step
    // rather to completion page
    if ($status === 'error' || empty($sk)) {
      $payment->delete();
      $this->logger->warning('Payment failed or cancelled for order @order_id, requestId: @request_id', [
        '@order_id' => $order_id,
        '@request_id' => $remote_id,
      ]);
      throw new PaymentGatewayException("Payment failed or cancelled");
    }

    // CRITICAL SECURITY: Verify SK (Security Key) to prevent fraud
    // According to Ifthenpay API: SK = SHA-256(orderId + amount + requestId + CCARD_KEY)
    $expected_hash = hash('sha256', $order_id . number_format($order_amount, 2, '.', '') . $remote_id . $this->configuration['cccard_key']);

    if ($sk !== $expected_hash) {
      $payment->delete();
      $this->logger->error('Security key validation failed for order @order_id. Expected: @expected, Got: @received', [
        '@order_id' => $order_id,
        '@expected' => substr($expected_hash, 0, 8) . '...',
        '@received' => substr($sk, 0, 8) . '...',
      ]);
      throw new PaymentGatewayException('Security validation failed');
    }

    // Verify amounts match (with proper decimal formatting)
    $formatted_charged_amount = number_format((float) $charged_amount, 2, '.', '');
    $formatted_order_amount = number_format((float) $order_amount, 2, '.', '');

    if ($formatted_charged_amount !== $formatted_order_amount) {
      $payment->delete();
      $this->logger->warning('Amount mismatch for order @order_id. Charged: @charged, Expected: @expected', [
        '@order_id' => $order_id,
        '@charged' => $formatted_charged_amount,
        '@expected' => $formatted_order_amount,
      ]);
      throw new PaymentGatewayException('Amount mismatch detected');
    }

    // All validations passed - complete the payment
    $payment->setState('completed')->save();

    $this->logger->info('Payment completed successfully for order @order_id, amount: @amount, requestId: @request_id', [
      '@order_id' => $order_id,
      '@amount' => $formatted_order_amount,
      '@request_id' => $remote_id,
    ]);

    $this->messenger()->addMessage($this->t('Payment was processed successfully'));
  }

  /**
   * {@inheritdoc}
   */
  public function onCancel(OrderInterface $order, Request $request) {
    parent::onCancel($order, $request);
  }


}
