<?php

namespace Drupal\commerce_ifthenpay\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\PaymentGatewayBase;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\ManualPaymentGatewayInterface;
use Drupal\commerce_payment\Plugin\Commerce\PaymentGateway\SupportsNotificationsInterface;
use Drupal\commerce_price\Price;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Response;
use InvalidArgumentException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the Ifthenpay payment gateway.
 *
 * @CommercePaymentGateway(
 *   id = "ifthenpay",
 *   label = "Ifthenpay - Multibanco",
 *   display_label = "Multibanco",
 *   modes = {
 *     "n/a" = @Translation("N/A"),
 *   },
 *   forms = {
 *     "add-payment" =
 *   "Drupal\commerce_ifthenpay\PluginForm\ManualPaymentAddForm",
 *     "receive-payment" =
 *   "Drupal\commerce_ifthenpay\PluginForm\PaymentReceiveForm",
 *   },
 *   requires_billing_information = FALSE,
 *   payment_type = "payment_manual",
 * )
 */
class Ifthenpay extends PaymentGatewayBase implements ManualPaymentGatewayInterface, SupportsNotificationsInterface {

  /**
   * Minimum allowed payment amount.
   */
  const MIN_AMOUNT = 1;

  /**
   * Maximum allowed payment amount.
   */
  const MAX_AMOUNT = 1000000;

  /**
   * Maximum order ID digits supported by Multibanco reference generation.
   */
  const MAX_ORDER_ID_DIGITS = 4;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected $logger;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->logger = $container->get('logger.channel.commerce_ifthenpay');
    return $instance;
  }

  /**
   * Gets the logger service.
   *
   * @return \Drupal\Core\Logger\LoggerChannelInterface
   *   The logger service.
   */
  protected function getLogger() {
    if (!$this->logger) {
      $this->logger = \Drupal::service('logger.channel.commerce_ifthenpay');
    }
    return $this->logger;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
        'instructions' => [
          'value' => '',
          'format' => 'plain_text',
        ],
      ] + parent::defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    $form = parent::buildConfigurationForm($form, $form_state);

    $form['multibanco_entidade'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Multibanco - Entidade'),
      '#description' => $this->t('The entity identifier provided by Ifthenpay (5 digits).'),
      '#default_value' => $this->configuration['multibanco_entidade'] ?? '',
      '#required' => TRUE,
      '#maxlength' => 5,
      '#size' => 10,
    ];
    $form['multibanco_subentidade'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Multibanco - Subentidade'),
      '#description' => $this->t('The sub-entity identifier provided by Ifthenpay (3 digits).'),
      '#default_value' => $this->configuration['multibanco_subentidade'] ?? '',
      '#required' => TRUE,
      '#maxlength' => 3,
      '#size' => 10,
    ];
    $form['multibanco_chaveAntiPhishing'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Chave AntiPhishing'),
      '#description' => $this->t('The anti-phishing key provided by Ifthenpay for webhook security.'),
      '#default_value' => $this->configuration['multibanco_chaveAntiPhishing'] ?? '',
      '#required' => TRUE,
      '#size' => 50,
    ];

    $form['instructions'] = [
      '#type' => 'text_format',
      '#title' => $this->t('Payment instructions'),
      '#description' => $this->t('Shown the end of checkout, after the customer has placed their order.'),
      '#default_value' => $this->configuration['instructions']['value'],
      '#format' => $this->configuration['instructions']['format'],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::validateConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);

      // Validate entity ID format (5 digits).
      if (!empty($values['multibanco_entidade']) && !preg_match('/^\d{5}$/', $values['multibanco_entidade'])) {
        $form_state->setError($form['multibanco_entidade'], $this->t('Entity ID must be exactly 5 digits.'));
      }

      // Validate sub-entity ID format (3 digits).
      if (!empty($values['multibanco_subentidade']) && !preg_match('/^\d{3}$/', $values['multibanco_subentidade'])) {
        $form_state->setError($form['multibanco_subentidade'], $this->t('Sub-entity ID must be exactly 3 digits.'));
      }

      // Validate anti-phishing key is not empty.
      if (empty($values['multibanco_chaveAntiPhishing'])) {
        $form_state->setError($form['multibanco_chaveAntiPhishing'], $this->t('Anti-phishing key is required for secure webhook processing.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    parent::submitConfigurationForm($form, $form_state);

    if (!$form_state->getErrors()) {
      $values = $form_state->getValue($form['#parents']);
      $this->configuration['instructions'] = $values['instructions'];
      $this->configuration['multibanco_entidade'] = $values['multibanco_entidade'];
      $this->configuration['multibanco_subentidade'] = $values['multibanco_subentidade'];
      $this->configuration['multibanco_chaveAntiPhishing'] = $values['multibanco_chaveAntiPhishing'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function buildPaymentInstructions(PaymentInterface $payment) {
    $instructions = [];
    $order_id = $payment->getOrderId();
    $amount = $payment->getAmount();
    $order_value = $amount->getNumber();

    // Check if this is an existing payment with a remote ID already set
    // If so, we should NOT regenerate or modify anything - just display instructions
    $existing_remote_id = $payment->getRemoteId();
    $is_existing_payment = !empty($existing_remote_id);

    try {
      if ($is_existing_payment) {
        // For existing payments, use the stored remote ID to reconstruct the reference
        // Format: "001234567" -> "001 234 567"
        $clean_remote_id = $existing_remote_id;
        $mb_ref = substr($clean_remote_id, 0, 3) . ' ' . substr($clean_remote_id, 3, 3) . ' ' . substr($clean_remote_id, 6);
      }
      else {
        // For new payments, generate the reference
        $mb_ref = self::generateMbRef(
          $this->configuration['multibanco_entidade'],
          $this->configuration['multibanco_subentidade'],
          $order_id,
          $order_value
        );
      }

      if (!empty($this->configuration['instructions']['value'])) {
        $instructions['intro'] = [
          '#type' => 'processed_text',
          '#text' => $this->configuration['instructions']['value'],
          '#format' => $this->configuration['instructions']['format'],
        ];
      }

      $instructions['mb_ref'] = [
        '#theme' => 'multibanco_instructions',
        '#entity' => $this->configuration['multibanco_entidade'],
        '#reference' => $mb_ref,
        '#amount' => $amount,
      ];

      // Only process new payments - don't modify existing ones
      if (!$is_existing_payment) {
        // Set the Multibanco Referência as the Payment Remote ID
        $clean_reference = str_replace(' ', '', $mb_ref);
        $payment->setRemoteId($clean_reference);

        // Store original order ID for audit trail and debugging purposes
        // This is necessary because:
        // 1. Webhook processing finds payments by remote ID (mapped reference)
        // 2. We need to trace back from mapped reference ID (e.g., 7834) to original ID (e.g., 12345)
        // 3. Provides context for collision detection and troubleshooting
        // 4. Enables detailed logging showing both original and mapped reference IDs
        // 5. Essential for debugging payment issues in production environments
        if ($order_id > 9999) {
          $order = $payment->getOrder();
          $order->setData('ifthenpay_original_order_id', $order_id);
          $order->save();
        }

        $payment->save();

        $this->getLogger()->info('Generated Multibanco reference @reference for order @order_id (new payment)', [
          '@reference' => $mb_ref,
          '@order_id' => $order_id,
        ]);
      }
    }
    catch (\Exception $e) {
      $this->getLogger()->error('Failed to generate Multibanco reference for order @order_id: @message', [
        '@order_id' => $order_id,
        '@message' => $e->getMessage(),
      ]);
      throw $e;
    }

    return $instructions;
  }

  /**
   * {@inheritdoc}
   */
  public function buildPaymentOperations(PaymentInterface $payment) {
    $payment_state = $payment->getState()->value;
    $operations = [];
    $operations['receive'] = [
      'title' => $this->t('Receive'),
      'page_title' => $this->t('Receive payment'),
      'plugin_form' => 'receive-payment',
      'access' => $payment_state == 'pending',
    ];
    $operations['void'] = [
      'title' => $this->t('Void'),
      'page_title' => $this->t('Void payment'),
      'plugin_form' => 'void-payment',
      'access' => $payment_state == 'pending',
    ];
    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  public function createPayment(PaymentInterface $payment, $received = FALSE) {
    $this->assertPaymentState($payment, ['new']);

    $payment->state = $received ? 'completed' : 'pending';
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function receivePayment(PaymentInterface $payment, Price $amount = NULL) {
    $this->assertPaymentState($payment, ['pending']);

    // If not specified, use the entire amount.
    $amount = $amount ?: $payment->getAmount();
    $payment->state = 'completed';
    $payment->setAmount($amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function voidPayment(PaymentInterface $payment) {
    $payment->state = 'voided';
    $payment->save();
  }

  /**
   * {@inheritdoc}
   */
  public function refundPayment(PaymentInterface $payment, Price $amount = NULL) {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);
    // If not specified, refund the entire amount.
    $amount = $amount ?: $payment->getAmount();
    $this->assertRefundAmount($payment, $amount);

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->state = 'partially_refunded';
    }
    else {
      $payment->state = 'refunded';
    }

    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

  /**
   * Generates a Multibanco reference code.
   *
   * This function implements the Ifthenpay algorithm for generating 9-digit
   * Multibanco reference codes with check digits for payment validation.
   *
   * IMPORTANT LIMITATION: Due to Multibanco banking system constraints, only
   * the rightmost 4 digits of the order ID are used in reference generation.
   * This means:
   * - Order IDs > 9999 will be truncated (e.g., order 12345 becomes 2345)
   * - This can cause duplicate references for different orders
   * - High-volume sites (>9999 orders) may experience payment mismatching
   *
   * Consider implementing custom order numbering or alternative payment methods
   * for high-volume e-commerce sites to avoid this limitation.
   *
   * @param string $ent_id
   *   The entity ID (5 digits).
   * @param string $subent_id
   *   The sub-entity ID (3 digits).
   * @param int $order_id
   *   The order ID (will be truncated to 4 digits).
   * @param float $order_value
   *   The order value in euros.
   *
   * @return string
   *   The formatted Multibanco reference.
   *
   * @throws \InvalidArgumentException
   *   When the order value is outside the valid range.
   */
  public static function generateMbRef($ent_id, $subent_id, $order_id, $order_value) {
    $chk_val = 0;
    $original_order_id = $order_id;

    $order_value = sprintf("%01.2f", $order_value);

    // Handle large order IDs with improved algorithm
    $processed_order_id = self::processOrderIdForReference($order_id);

    // Log warning if order ID was modified
    if ($processed_order_id != $order_id) {
      \Drupal::logger('commerce_ifthenpay')->warning(
        'Order ID @original was converted to @processed for Multibanco reference generation. Original order: @original, Reference order: @processed',
        [
          '@original' => $original_order_id,
          '@processed' => $processed_order_id,
        ]
      );

      // Also log to Commerce Log if available
      self::logOrderIdConversion($original_order_id, $processed_order_id);
    }


    if ($order_value < self::MIN_AMOUNT || $order_value >= self::MAX_AMOUNT) {
      throw new InvalidArgumentException(sprintf(
        'Order amount must be between %d and %d euros. Got: %s',
        self::MIN_AMOUNT,
        self::MAX_AMOUNT - 1,
        $order_value
      ));
    }

    // Calculate check digits using Ifthenpay algorithm
    $chk_str = sprintf('%05u%03u%04u%08u', $ent_id, $subent_id, $processed_order_id, round($order_value * 100));

    $chk_array = [
      3,
      30,
      9,
      90,
      27,
      76,
      81,
      34,
      49,
      5,
      50,
      15,
      53,
      45,
      62,
      38,
      89,
      17,
      73,
      51,
    ];

    for ($i = 0; $i < 20; $i++) {
      $chk_int = substr($chk_str, 19 - $i, 1);
      $chk_val += ($chk_int % 10) * $chk_array[$i];
    }

    $chk_val %= 97;

    $chk_digits = sprintf('%02u', 98 - $chk_val);

    return $subent_id . ' ' . substr($chk_str, 8, 3) . ' ' . substr($chk_str, 11, 1) . $chk_digits;
  }

  /**
   * {@inheritdoc}
   */
  public function onNotify(Request $request) {
    $referencia = $request->get('referencia');
    $chave = $request->get('chave');
    $valor = $request->get('valor');
    $entidade = $request->get('entidade');

    // Validate required parameters
    if (empty($referencia) || empty($chave) || empty($valor) || empty($entidade)) {
      $this->getLogger()->warning('Invalid notification: missing required parameters. Request data: @data', [
        '@data' => json_encode($request->query->all()),
      ]);
      return new Response('Missing required parameters', 400);
    }

    // Validate entity ID
    $configured_entity = $this->configuration['multibanco_entidade'] ?? '';
    if (!empty($configured_entity) && $configured_entity !== $entidade) {
      $this->getLogger()->warning('Invalid entity ID for reference: @reference. Expected: @expected, Got: @received', [
        '@reference' => $referencia,
        '@expected' => $configured_entity,
        '@received' => $entidade,
      ]);
      return new Response('Invalid entity', 403);
    }

    // Validate anti-phishing key
    $configured_key = $this->configuration['multibanco_chaveAntiPhishing'] ?? '';
    if (!empty($configured_key) && $configured_key !== $chave) {
      $this->getLogger()->warning('Invalid anti-phishing key for reference: @reference. Expected: @expected, Got: @received', [
        '@reference' => $referencia,
        '@expected' => substr($configured_key, 0, 4) . '****',
        '@received' => substr($chave, 0, 4) . '****',
      ]);
      return new Response('Invalid authentication', 403);
    }

    try {
      // COLLISION-AWARE PAYMENT LOADING:
      // Instead of using loadByRemoteId() which only returns the first match,
      // we search for ALL payments with this remote ID to handle collisions properly
      $candidate_payments = $this->findAllPaymentsByRemoteId($referencia);

      if (!empty($candidate_payments)) {
        // Find the best matching payment by amount and state
        $payment = $this->selectBestPaymentCandidate($candidate_payments, (float) $valor, $referencia);

        if ($payment) {
          // Payment found and validated through collision-aware selection
          $payment_amount = $payment->getAmount();
          $callback_amount = (float) $valor;
          $payment_amount_number = (float) $payment_amount->getNumber();

          // Final amount validation (should already match due to selection logic)
          if (abs($payment_amount_number - $callback_amount) > 0.01) {
            // This should rarely happen since selectBestPaymentCandidate already validates amounts
            $this->getLogger()->error('CRITICAL: Amount mismatch after collision-aware selection! Reference: @reference, Expected: @expected, Got: @got (Order: @order_id)', [
              '@reference' => $referencia,
              '@expected' => $payment_amount->__toString(),
              '@got' => $callback_amount,
              '@order_id' => $payment->getOrderId(),
            ]);

            // Log critical error to Commerce Log
            try {
              if (\Drupal::moduleHandler()->moduleExists('commerce_log')) {
                $order = $payment->getOrder();
                if ($order) {
                  /** @var \Drupal\commerce_log\LogStorageInterface $log_storage */
                  $log_storage = \Drupal::entityTypeManager()->getStorage('commerce_log');
                  $log = $log_storage->generate($order, 'ifthenpay_amount_mismatch', [
                    'reference' => $referencia,
                    'payment_amount' => $payment_amount->__toString(),
                    'callback_amount' => $callback_amount,
                  ]);
                  $log->save();
                }
              }
            }
            catch (\Exception $e) {
              // Continue processing even if Commerce Log fails
            }

            return new Response('Critical amount validation failure', 500);
          }

          // Only update if payment is still pending
          if ($payment->getState()->getId() === 'pending') {
            $payment->setState('completed')->save();

            // CRITICAL: Explicitly save the order to trigger commerce_order.order.paid event.
            //
            // Commerce core normally handles this via PaymentOrderUpdater service, which
            // defers order updates until the end of the request (DestructableInterface).
            // However, this mechanism can fail in webhook/async contexts due to:
            // - Early termination or exceptions before destruct is called
            // - Memory limits or timeouts in long-running processes
            // - Service container issues in queue processors
            //
            // By explicitly saving the order here when balance is zero, we ensure the
            // commerce_order.order.paid event fires reliably via OrderStorage::doOrderPreSave(),
            // which is critical for downstream processes (ERP sync, notifications, etc).
            //
            // This approach is more reliable for webhook handlers than relying on destruct.
            $order = $payment->getOrder();
            if ($order && ($order->getBalance()->isZero() || $order->getBalance()->isNegative())) {
              $order->save();
            }

            $this->getLogger()->info('Payment completed for reference: @reference (Order: @order_id, Amount: @amount, Verified: @callback_amount)', [
              '@reference' => $referencia,
              '@order_id' => $payment->getOrderId(),
              '@amount' => $payment->getAmount()->__toString(),
              '@callback_amount' => $callback_amount,
            ]);
          }
          else {
            $this->getLogger()->info('Payment already processed for reference: @reference (Order: @order_id, state: @state, Amount verified: @callback_amount)', [
              '@reference' => $referencia,
              '@order_id' => $payment->getOrderId(),
              '@state' => $payment->getState()->getId(),
              '@callback_amount' => $callback_amount,
            ]);
          }
        }
        else {
          $this->getLogger()->warning('No valid payment found for reference: @reference after collision-aware search', [
            '@reference' => $referencia,
          ]);
          return new Response('Payment not found', 404);
        }
      }
    }
    catch (\Exception $e) {
      $this->getLogger()->error('Error processing notification for reference @reference: @message. Stack trace: @trace', [
        '@reference' => $referencia,
        '@message' => $e->getMessage(),
        '@trace' => $e->getTraceAsString(),
      ]);
      return new Response('Internal server error', 500);
    }

    return new Response('OK', 200);
  }

  /**
   * {@inheritdoc}
   */
  public function getNotifyUrl() {
    return Url::fromRoute('commerce_payment.notify', [
      'commerce_payment_gateway' => $this->parentEntity->id(),
    ], [
      'absolute' => TRUE,
    ]);
  }

  /**
   * Gets the configured entity ID.
   *
   * @return string
   *   The entity ID.
   */
  protected function getEntityId(): string {
    return $this->configuration['multibanco_entidade'] ?? '';
  }

  /**
   * Gets the configured sub-entity ID.
   *
   * @return string
   *   The sub-entity ID.
   */
  protected function getSubEntityId(): string {
    return $this->configuration['multibanco_subentidade'] ?? '';
  }

  /**
   * Gets the configured anti-phishing key.
   *
   * @return string
   *   The anti-phishing key.
   */
  protected function getAntiPhishingKey(): string {
    return $this->configuration['multibanco_chaveAntiPhishing'] ?? '';
  }

  /**
   * Processes order ID to fit within Multibanco 4-digit limitation.
   *
   * COLLISION PREVENTION STRATEGY:
   * ===============================
   * The Multibanco banking system constrains references to exactly 4 digits for order IDs.
   * This creates a fundamental problem: unlimited order IDs must fit into 9000 possible
   * values (1000-9999). Our multi-layered approach minimizes collision risk:
   *
   * LAYER 1 - SMART ALGORITHM:
   * - Order IDs <= 9999: Use directly (backward compatible, zero collision risk)
   * - Order IDs > 9999: Apply hash-based mapping instead of simple truncation
   * - Hash function (CRC32) creates pseudo-random distribution across 4-digit space
   * - Formula: (abs(crc32(order_id)) % 9000) + 1000 ensures range 1000-9999
   *
   * LAYER 2 - COLLISION DETECTION:
   * - Proactively check for existing payments with same mapped reference ID
   * - Log warnings when potential collisions are detected
   * - Provides early warning system for administrators
   *
   * LAYER 3 - COMPREHENSIVE LOGGING:
   * - System logs: All conversions and collisions logged to watchdog
   * - Commerce logs: Order-specific warnings appear in order management
   * - Audit trail: Original and mapped reference IDs tracked for debugging
   *
   * LAYER 4 - DATA PRESERVATION:
   * - Original order IDs stored in order data for large orders
   * - Enables traceability from mapped reference ID back to original
   * - Critical for webhook processing and troubleshooting
   *
   * MATHEMATICAL ANALYSIS:
   * - Simple truncation: ~100% collision rate for similar order IDs
   * - Hash-based approach: ~0.011% collision rate for random orders
   * - Birthday paradox: 50% collision chance after ~112 large orders
   * - 90% collision chance after ~224 large orders
   *
   * This approach transforms a guaranteed collision problem into a manageable
   * risk with comprehensive monitoring and early detection capabilities.
   *
   * @param int $order_id
   *   The original order ID.
   *
   * @return int
   *   A 4-digit mapped reference ID suitable for Multibanco reference generation.
   */
  protected static function processOrderIdForReference(int $order_id): int {
    // If order ID fits in 4 digits, use it directly (backward compatible)
    if ($order_id <= 9999) {
      return $order_id;
    }

    // HASH-BASED TRANSFORMATION ALGORITHM:
    // ====================================
    // Instead of simple truncation (e.g., 12345 -> 2345), we use a hash function
    // to create pseudo-random distribution across the 4-digit space.
    //
    // Step 1: Generate hash from order ID string
    // CRC32 produces a 32-bit signed integer (-2^31 to +2^31-1)
    $hash = crc32((string) $order_id);

    // Step 2: Convert to positive and map to 4-digit range (1000-9999)
    // - abs(): Handle negative hash values
    // - % 9000: Map to range 0-8999 (exactly 9000 possible values)
    // - + 1000: Shift to range 1000-9999 (ensures 4-digit results)
    // This gives us even distribution across all possible 4-digit order IDs
    $processed_id = (abs($hash) % 9000) + 1000;

    // Check for potential collisions with existing payments
    self::checkForOrderIdCollisions($order_id, $processed_id);

    return $processed_id;
  }

  /**
   * Checks for potential order ID collisions and logs warnings.
   *
   * @param int $original_order_id
   *   The original order ID.
   * @param int $processed_order_id
   *   The processed 4-digit order ID.
   */
  protected static function checkForOrderIdCollisions(int $original_order_id, int $processed_order_id): void {
    // Query for existing payments with the same processed order ID
    $payment_storage = \Drupal::entityTypeManager()->getStorage('commerce_payment');
    $query = $payment_storage->getQuery()
      ->condition('remote_id', '%' . str_pad($processed_order_id, 4, '0', STR_PAD_LEFT) . '%', 'LIKE')
      ->range(0, 1)
      ->accessCheck(FALSE);

    $existing_payments = $query->execute();

    if (!empty($existing_payments)) {
      $warning_message = sprintf(
        'Potential order ID collision detected! Order %d maps to %d, but payments with similar references already exist. This may cause payment mismatching.',
        $original_order_id,
        $processed_order_id
      );

      // Log to system logger
      \Drupal::logger('commerce_ifthenpay')->warning($warning_message);

      // Also log to the current order if Commerce Log module is available
      try {
        if (\Drupal::moduleHandler()->moduleExists('commerce_log')) {
          $order_storage = \Drupal::entityTypeManager()->getStorage('commerce_order');
          $order = $order_storage->load($original_order_id);

          if ($order) {
            /** @var \Drupal\commerce_log\LogStorageInterface $log_storage */
            $log_storage = \Drupal::entityTypeManager()->getStorage('commerce_log');

            // Create a custom log entry for the collision warning
            // Note: This requires defining a custom log template in the module's .yml files
            $log = $log_storage->generate($order, 'ifthenpay_collision_warning', [
              'original_order_id' => $original_order_id,
              'processed_order_id' => $processed_order_id,
              'warning_message' => $warning_message,
            ]);
            $log->save();

            \Drupal::logger('commerce_ifthenpay')->info(
              'Collision warning logged to Commerce order @order_id (Log ID: @log_id)',
              [
                '@order_id' => $original_order_id,
                '@log_id' => $log->id(),
              ]
            );
          }
        }
        else {
          \Drupal::logger('commerce_ifthenpay')->info(
            'Commerce Log module not available. Collision warning for order @order_id logged to system only.',
            ['@order_id' => $original_order_id]
          );
        }
      }
      catch (\Exception $e) {
        // If we can't log to the order, just continue
        \Drupal::logger('commerce_ifthenpay')->debug(
          'Could not log collision warning to order @order_id: @error',
          [
            '@order_id' => $original_order_id,
            '@error' => $e->getMessage(),
          ]
        );
      }
    }
  }

  /**
   * Finds ALL payments with the specified remote ID.
   *
   * Unlike PaymentStorage::loadByRemoteId() which only returns the first match,
   * this method returns all payments that have the same remote ID. This is
   * essential for handling order ID collisions where multiple payments might
   * share the same Multibanco reference.
   *
   * @param string $remote_id
   *   The remote ID (Multibanco reference) to search for.
   *
   * @return \Drupal\commerce_payment\Entity\PaymentInterface[]
   *   Array of payment entities with the specified remote ID.
   */
  protected function findAllPaymentsByRemoteId(string $remote_id): array {
    try {
      /** @var \Drupal\commerce_payment\PaymentStorage $payment_storage */
      $payment_storage = $this->entityTypeManager->getStorage('commerce_payment');

      $query = $payment_storage->getQuery()
        ->condition('payment_gateway', $this->parentEntity->id())
        ->condition('remote_id', $remote_id)
        ->accessCheck(FALSE);

      $payment_ids = $query->execute();

      if (!empty($payment_ids)) {
        return $payment_storage->loadMultiple($payment_ids);
      }

      return [];
    }
    catch (\Exception $e) {
      $this->getLogger()->error('Error searching for payments by remote ID @remote_id: @error', [
        '@remote_id' => $remote_id,
        '@error' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Selects the best payment candidate from multiple payments with the same remote ID.
   *
   * When multiple payments share the same remote ID (collision scenario), this method
   * applies intelligent selection logic to find the most appropriate payment to process:
   *
   * SELECTION PRIORITY:
   * 1. Exact amount match + pending state (highest priority)
   * 2. Exact amount match + any state (if no pending found)
   * 3. Log collision details for manual review
   *
   * @param \Drupal\commerce_payment\Entity\PaymentInterface[] $candidate_payments
   *   Array of payment candidates with the same remote ID.
   * @param float $callback_amount
   *   The payment amount from the callback.
   * @param string $referencia
   *   The Multibanco reference for logging purposes.
   *
   * @return \Drupal\commerce_payment\Entity\PaymentInterface|null
   *   The best matching payment or null if no suitable candidate found.
   */
  protected function selectBestPaymentCandidate(array $candidate_payments, float $callback_amount, string $referencia): ?PaymentInterface {
    $pending_matches = [];
    $completed_matches = [];
    $amount_mismatches = [];

    // Analyze all candidates
    foreach ($candidate_payments as $candidate) {
      /** @var \Drupal\commerce_payment\Entity\PaymentInterface $candidate */
      $candidate_amount = (float) $candidate->getAmount()->getNumber();
      $amount_matches = abs($candidate_amount - $callback_amount) <= 0.01;

      if ($amount_matches) {
        if ($candidate->getState()->getId() === 'pending') {
          $pending_matches[] = $candidate;
        } else {
          $completed_matches[] = $candidate;
        }
      } else {
        $amount_mismatches[] = $candidate;
      }
    }

    // Log collision details for transparency
    if (count($candidate_payments) > 1) {
      $this->getLogger()->warning('Multiple payments found for reference @reference. Candidates: @candidates', [
        '@reference' => $referencia,
        '@candidates' => $this->formatCandidatesForLog($candidate_payments, $callback_amount),
      ]);
    }

    // Selection logic: prefer pending payments with exact amount match
    if (!empty($pending_matches)) {
      $selected = reset($pending_matches);
      $this->getLogger()->info('Selected pending payment with matching amount: Order @order_id, Amount @amount', [
        '@order_id' => $selected->getOrderId(),
        '@amount' => $selected->getAmount()->__toString(),
      ]);
      return $selected;
    }

    // Fallback: completed payment with exact amount match
    if (!empty($completed_matches)) {
      $selected = reset($completed_matches);
      $this->getLogger()->info('Selected completed payment with matching amount: Order @order_id, Amount @amount, State @state', [
        '@order_id' => $selected->getOrderId(),
        '@amount' => $selected->getAmount()->__toString(),
        '@state' => $selected->getState()->getId(),
      ]);
      return $selected;
    }

    // No amount matches found
    $this->getLogger()->error('No payment candidates with matching amount @amount found for reference @reference', [
      '@amount' => $callback_amount,
      '@reference' => $referencia,
    ]);

    return null;
  }

  /**
   * Formats payment candidates for logging purposes.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentInterface[] $candidates
   *   Array of payment candidates.
   * @param float $callback_amount
   *   The callback amount for comparison.
   *
   * @return string
   *   Formatted string describing all candidates.
   */
  protected function formatCandidatesForLog(array $candidates, float $callback_amount): string {
    $formatted = [];
    foreach ($candidates as $candidate) {
      $amount = (float) $candidate->getAmount()->getNumber();
      $match_status = abs($amount - $callback_amount) <= 0.01 ? 'MATCH' : 'MISMATCH';
      $formatted[] = sprintf('Order %d (€%.2f, %s, %s)',
        $candidate->getOrderId(),
        $amount,
        $candidate->getState()->getId(),
        $match_status
      );
    }
    return implode('; ', $formatted);
  }

  /**
   * Logs order ID conversion to Commerce Log if available.
   *
   * @param int $original_order_id
   *   The original order ID.
   * @param int $processed_order_id
   *   The processed 4-digit order ID.
   */
  protected static function logOrderIdConversion(int $original_order_id, int $processed_order_id): void {
    try {
      if (\Drupal::moduleHandler()->moduleExists('commerce_log')) {
        $order_storage = \Drupal::entityTypeManager()->getStorage('commerce_order');
        $order = $order_storage->load($original_order_id);

        if ($order) {
          /** @var \Drupal\commerce_log\LogStorageInterface $log_storage */
          $log_storage = \Drupal::entityTypeManager()->getStorage('commerce_log');

          $log = $log_storage->generate($order, 'ifthenpay_order_id_conversion', [
            'original_order_id' => $original_order_id,
            'processed_order_id' => $processed_order_id,
          ]);
          $log->save();
        }
      }
    }
    catch (\Exception $e) {
      // Silently fail if we can't log to Commerce Log
      \Drupal::logger('commerce_ifthenpay')->debug(
        'Could not log order ID conversion to Commerce Log: @error',
        ['@error' => $e->getMessage()]
      );
    }
  }

}
