<?php

namespace Drupal\commerce_ingenico\Plugin\Commerce\PaymentGateway;

use Drupal\commerce_payment\Entity\PaymentInterface;
use Drupal\commerce_payment\Exception\DeclineException;
use Drupal\commerce_payment\Exception\InvalidResponseException;
use Drupal\commerce_price\Price;
use Ogone\DirectLink\DirectLinkMaintenanceRequest;
use Ogone\DirectLink\DirectLinkMaintenanceResponse;
use Ogone\DirectLink\MaintenanceOperation;
use Ogone\HashAlgorithm;
use Ogone\Passphrase;
use Ogone\ShaComposer\AllParametersShaComposer;

trait OperationsTrait {

  /**
   * {@inheritdoc}
   */
  protected function getDefaultForms() {
    $default_forms = parent::getDefaultForms();

    $default_forms['renew-payment-authorization'] = 'Drupal\commerce_ingenico\PluginForm\PaymentRenewAuthorizationForm';

    return $default_forms;
  }

  /**
   * {@inheritdoc}
   */
  public function buildPaymentOperations(PaymentInterface $payment) {
    $operations = parent::buildPaymentOperations($payment);

    $access = $payment->getState()->value == 'authorization';
    $operations['renew_authorization'] = [
      'title' => $this->t('Renew authorization'),
      'page_title' => $this->t('Renew payment authorization'),
      'plugin_form' => 'renew-payment-authorization',
      'access' => $access,
      'weight' => 10,
    ];

    return $operations;
  }

  /**
   * {@inheritdoc}
   *
   * @see https://payment-services.ingenico.com/int/en/ogone/support/guides/integration%20guides/directlink/maintenance
   */
  public function capturePayment(PaymentInterface $payment, Price $amount = NULL) {
    $this->assertPaymentState($payment, ['authorization']);

    // If not specified, capture the entire amount.
    $amount = $amount ?: $payment->getAmount();

    // Validate the requested amount.
    $this->assertRefundAmount($payment, $amount);

    $passphrase = new Passphrase($this->configuration['sha_in']);
    $sha_algorithm = new HashAlgorithm($this->configuration['sha_algorithm']);
    $shaComposer = new AllParametersShaComposer($passphrase, $sha_algorithm);

    $directLinkRequest = new DirectLinkMaintenanceRequest($shaComposer);

    $directLinkRequest->setPspid($this->configuration['pspid']);
    $directLinkRequest->setUserId($this->configuration['userid']);
    $directLinkRequest->setPassword($this->configuration['password']);
    $directLinkRequest->setPayId($payment->getRemoteId());
    // Ingenico requires the AMOUNT value to be sent in decimals.
    $directLinkRequest->setAmount((int) $this->minorUnitsConverter->toMinorUnits($amount));

    $operation = $balance->subtract($amount)->isZero() ? MaintenanceOperation::OPERATION_CAPTURE_LAST_OR_FULL : MaintenanceOperation::OPERATION_CAPTURE_PARTIAL;
    $directLinkRequest->setOperation(new MaintenanceOperation($operation));

    $directLinkRequest->validate();

    // We cannot use magic set method to AbstractRequest::__call the SHASIGN
    // value (as it is not on the list of Ogone fields), so let's get all
    // already set parameters, and add SHASIGN to them here.
    $body = $directLinkRequest->toArray();
    $body['SHASIGN'] = $directLinkRequest->getShaSign();

    // Log the request message if request logging is enabled.
    if (!empty($this->configuration['api_logging']['request'])) {
      \Drupal::logger('commerce_ingenico')
        ->debug('DirectLink capture request: @url <pre>@body</pre>', [
          '@url' => $this->getOgoneUri($directLinkRequest),
          '@body' => var_export($body, TRUE),
        ]);
    }

    // Perform the request to Ingenico API.
    $options = [
      'form_params' => $body,
    ];
    $response = $this->httpClient->post($this->getOgoneUri($directLinkRequest), $options);

    // Log the response message if request logging is enabled.
    if (!empty($this->configuration['api_logging']['response'])) {
      \Drupal::logger('commerce_ingenico')
        ->debug('DirectLink capture response: <pre>@body</pre>', [
          '@body' => (string) $response->getBody(),
        ]);
    }

    // Validate the API response.
    if ($response->getStatusCode() != 200) {
      throw new InvalidResponseException($this->t('The request returned with error code @http_code.', ['@http_code' => $response->getStatusCode()]));
    }
    elseif (!$response->getBody()) {
      throw new InvalidResponseException($this->t('The response did not have a body.'));
    }

    $directLinkResponse = new DirectLinkMaintenanceResponse($response->getBody());
    if (!$directLinkResponse->isSuccessful()) {
      throw new DeclineException($this->t('Transaction has been declined by the gateway (@error_code).', [
        '@error_code' => $directLinkResponse->getParam('NCERROR'),
      ]), (int) $directLinkResponse->getParam('NCERROR'));
    }

    $payment->state = 'completed';
    $payment->setRemoteState($directLinkResponse->getParam('STATUS'));
    $payment->setAmount($amount);
    $payment->save();
  }

  /**
   * {@inheritdoc}
   *
   * @see https://payment-services.ingenico.com/int/en/ogone/support/guides/integration%20guides/directlink/maintenance
   */
  public function voidPayment(PaymentInterface $payment) {
    $this->assertPaymentState($payment, ['authorization']);

    $passphrase = new Passphrase($this->configuration['sha_in']);
    $sha_algorithm = new HashAlgorithm($this->configuration['sha_algorithm']);
    $shaComposer = new AllParametersShaComposer($passphrase, $sha_algorithm);

    $directLinkRequest = new DirectLinkMaintenanceRequest($shaComposer);

    $directLinkRequest->setPspid($this->configuration['pspid']);
    $directLinkRequest->setUserId($this->configuration['userid']);
    $directLinkRequest->setPassword($this->configuration['password']);
    $directLinkRequest->setPayId($payment->getRemoteId());

    $operation = MaintenanceOperation::OPERATION_AUTHORISATION_DELETE_AND_CLOSE;
    $directLinkRequest->setOperation(new MaintenanceOperation($operation));

    $directLinkRequest->validate();

    // We cannot use magic set method to AbstractRequest::__call the SHASIGN
    // value (as it is not on the list of Ogone fields), so let's get all
    // already set parameters, and add SHASIGN to them here.
    $body = $directLinkRequest->toArray();
    $body['SHASIGN'] = $directLinkRequest->getShaSign();

    // Log the request message if request logging is enabled.
    if (!empty($this->configuration['api_logging']['request'])) {
      \Drupal::logger('commerce_ingenico')
        ->debug('DirectLink void request: @url <pre>@body</pre>', [
          '@url' => $this->getOgoneUri($directLinkRequest),
          '@body' => var_export($body, TRUE),
        ]);
    }

    // Perform the request to Ingenico API.
    $options = [
      'form_params' => $body,
    ];
    $response = $this->httpClient->post($this->getOgoneUri($directLinkRequest), $options);

    // Log the response message if request logging is enabled.
    if (!empty($this->configuration['api_logging']['response'])) {
      \Drupal::logger('commerce_ingenico')
        ->debug('DirectLink void response: <pre>@body</pre>', [
          '@body' => (string) $response->getBody(),
        ]);
    }

    // Validate the API response.
    if ($response->getStatusCode() != 200) {
      throw new InvalidResponseException($this->t('The request returned with error code @http_code.', ['@http_code' => $response->getStatusCode()]));
    }
    elseif (!$response->getBody()) {
      throw new InvalidResponseException($this->t('The response did not have a body.'));
    }

    $directLinkResponse = new DirectLinkMaintenanceResponse($response->getBody());
    if (!$directLinkResponse->isSuccessful()) {
      throw new DeclineException($this->t('Transaction has been declined by the gateway (@error_code).', [
        '@error_code' => $directLinkResponse->getParam('NCERROR'),
      ]), (int) $directLinkResponse->getParam('NCERROR'));
    }

    $payment->state = 'authorization_voided';
    $payment->setRemoteState($directLinkResponse->getParam('STATUS'));
    $payment->save();
  }

  /**
   * Renews the authorization for the given payment.
   *
   * Only authorizations for payments in the 'authorization' state can be
   * renewed.
   *
   * @param \Drupal\commerce_payment\Entity\PaymentInterface $payment
   *   The payment to renew the authorization for.
   *
   * @throws \InvalidArgumentException
   *   Thrown when the payment is not in the 'authorization' state.
   * @throws \Drupal\commerce_payment\Exception\InvalidResponseException
   *   Thrown when the invalid response is returned by the gateway.
   * @throws \Drupal\commerce_payment\Exception\DeclineException
   *   Thrown when the transaction is declined by the gateway.
   *
   * @see DirectLink::buildPaymentOperations()
   * @see DirectLink::getDefaultForms()
   *
   * @see https://payment-services.ingenico.com/int/en/ogone/support/guides/integration%20guides/directlink/maintenance
   */
  public function renewAuthorization(PaymentInterface $payment) {
    $this->assertPaymentState($payment, ['authorization']);

    $passphrase = new Passphrase($this->configuration['sha_in']);
    $sha_algorithm = new HashAlgorithm($this->configuration['sha_algorithm']);
    $shaComposer = new AllParametersShaComposer($passphrase, $sha_algorithm);

    $directLinkRequest = new DirectLinkMaintenanceRequest($shaComposer);

    $directLinkRequest->setPspid($this->configuration['pspid']);
    $directLinkRequest->setUserId($this->configuration['userid']);
    $directLinkRequest->setPassword($this->configuration['password']);
    $directLinkRequest->setPayId($payment->getRemoteId());

    $operation = MaintenanceOperation::OPERATION_AUTHORISATION_RENEW;
    $directLinkRequest->setOperation(new MaintenanceOperation($operation));

    $directLinkRequest->validate();

    // We cannot use magic set method to AbstractRequest::__call the SHASIGN
    // value (as it is not on the list of Ogone fields), so let's get all
    // already set parameters, and add SHASIGN to them here.
    $body = $directLinkRequest->toArray();
    $body['SHASIGN'] = $directLinkRequest->getShaSign();

    // Log the request message if request logging is enabled.
    if (!empty($this->configuration['api_logging']['request'])) {
      \Drupal::logger('commerce_ingenico')
        ->debug('DirectLink renew authorization request: @url <pre>@body</pre>', [
          '@url' => $this->getOgoneUri($directLinkRequest),
          '@body' => var_export($body, TRUE),
        ]);
    }

    // Perform the request to Ingenico API.
    $options = [
      'form_params' => $body,
    ];
    $response = $this->httpClient->post($this->getOgoneUri($directLinkRequest), $options);

    // Log the response message if request logging is enabled.
    if (!empty($this->configuration['api_logging']['response'])) {
      \Drupal::logger('commerce_ingenico')
        ->debug('DirectLink renew authorization response: <pre>@body</pre>', [
          '@body' => (string) $response->getBody(),
        ]);
    }

    // Validate the API response.
    if ($response->getStatusCode() != 200) {
      throw new InvalidResponseException($this->t('The request returned with error code @http_code.', ['@http_code' => $response->getStatusCode()]));
    }
    elseif (!$response->getBody()) {
      throw new InvalidResponseException($this->t('The response did not have a body.'));
    }

    $directLinkResponse = new DirectLinkMaintenanceResponse($response->getBody());
    if (!$directLinkResponse->isSuccessful()) {
      throw new DeclineException($this->t('Transaction has been declined by the gateway (@error_code).', [
        '@error_code' => $directLinkResponse->getParam('NCERROR'),
      ]), (int) $directLinkResponse->getParam('NCERROR'));
    }

    $payment->setRemoteState($directLinkResponse->getParam('STATUS'));
    $payment->save();
  }

  /**
   * {@inheritdoc}
   *
   * @see https://payment-services.ingenico.com/int/en/ogone/support/guides/integration%20guides/directlink/maintenance
   */
  public function refundPayment(PaymentInterface $payment, Price $amount = NULL) {
    $this->assertPaymentState($payment, ['completed', 'partially_refunded']);

    // If not specified, refund the entire amount.
    $amount = $amount ?: $payment->getAmount();

    // Validate the requested amount.
    $this->assertRefundAmount($payment, $amount);

    $passphrase = new Passphrase($this->configuration['sha_in']);
    $sha_algorithm = new HashAlgorithm($this->configuration['sha_algorithm']);
    $shaComposer = new AllParametersShaComposer($passphrase, $sha_algorithm);

    $directLinkRequest = new DirectLinkMaintenanceRequest($shaComposer);

    $directLinkRequest->setPspid($this->configuration['pspid']);
    $directLinkRequest->setUserId($this->configuration['userid']);
    $directLinkRequest->setPassword($this->configuration['password']);
    $directLinkRequest->setPayId($payment->getRemoteId());
    // Ingenico requires the AMOUNT value to be sent in decimals.
    $directLinkRequest->setAmount((int) $this->minorUnitsConverter->toMinorUnits($amount));
    $balance = $payment->getBalance();
    $operation = $balance->subtract($amount)->isZero() ? MaintenanceOperation::OPERATION_REFUND_LAST_OR_FULL : MaintenanceOperation::OPERATION_REFUND_PARTIAL;
    $directLinkRequest->setOperation(new MaintenanceOperation($operation));

    $directLinkRequest->validate();

    // We cannot use magic set method to AbstractRequest::__call the SHASIGN
    // value (as it is not on the list of Ogone fields), so let's get all
    // already set parameters, and add SHASIGN to them here.
    $body = $directLinkRequest->toArray();
    $body['SHASIGN'] = $directLinkRequest->getShaSign();

    // Log the request message if request logging is enabled.
    if (!empty($this->configuration['api_logging']['request'])) {
      \Drupal::logger('commerce_ingenico')
        ->debug('DirectLink refund request: @url <pre>@body</pre>', [
          '@url' => $this->getOgoneUri($directLinkRequest),
          '@body' => var_export($body, TRUE),
        ]);
    }

    // Perform the request to Ingenico API.
    $options = [
      'form_params' => $body,
    ];
    $response = $this->httpClient->post($this->getOgoneUri($directLinkRequest), $options);

    // Log the response message if request logging is enabled.
    if (!empty($this->configuration['api_logging']['response'])) {
      \Drupal::logger('commerce_ingenico')
        ->debug('DirectLink refund response: <pre>@body</pre>', [
          '@body' => (string) $response->getBody(),
        ]);
    }

    // Validate the API response.
    if ($response->getStatusCode() != 200) {
      throw new InvalidResponseException($this->t('The request returned with error code @http_code.', ['@http_code' => $response->getStatusCode()]));
    }
    elseif (!$response->getBody()) {
      throw new InvalidResponseException($this->t('The response did not have a body.'));
    }

    $directLinkResponse = new DirectLinkMaintenanceResponse($response->getBody());
    if (!$directLinkResponse->isSuccessful()) {
      throw new DeclineException($this->t('Transaction has been declined by the gateway (@error_code).', [
        '@error_code' => $directLinkResponse->getParam('NCERROR'),
      ]), (int) $directLinkResponse->getParam('NCERROR'));
    }

    $old_refunded_amount = $payment->getRefundedAmount();
    $new_refunded_amount = $old_refunded_amount->add($amount);
    if ($new_refunded_amount->lessThan($payment->getAmount())) {
      $payment->state = 'partially_refunded';
    }
    else {
      $payment->state = 'refunded';
    }

    $payment->setRemoteState($directLinkResponse->getParam('STATUS'));
    $payment->setRefundedAmount($new_refunded_amount);
    $payment->save();
  }

}
