<?php

namespace Drupal\commerce_irpaymentpack\Banks;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Handles Zarinpal Gateway transactions
 *
 * @author "Ahmad Hejazee" <mngafa@gmail.com>
 */
class Zarinpal {

  use StringTranslationTrait;

  protected string $merchantID;
  protected int $amount;

  /** REST API Endpoints **/
  protected const API_REQUEST_LIVE = 'https://api.zarinpal.com/pg/v4/payment/request.json';
  protected const API_VERIFY_LIVE = 'https://api.zarinpal.com/pg/v4/payment/verify.json';
  protected const API_REQUEST_SANDBOX = 'https://sandbox.zarinpal.com/pg/v4/payment/request.json';
  protected const API_VERIFY_SANDBOX = 'https://sandbox.zarinpal.com/pg/v4/payment/verify.json';
  protected const GATEWAY_URL = 'https://www.zarinpal.com/pg/StartPay/';
  protected const GATEWAY_URL_SANDBOX = 'https://sandbox.zarinpal.com/pg/StartPay/';

  protected bool $sandbox = FALSE;

  function __construct(string $merchantID, int $amount) {
    $this->merchantID = $merchantID;
    $this->amount = $amount;
  }

  public function getSandbox(): bool {
    return $this->sandbox;
  }

  public function setSandbox(bool $sandbox = TRUE) {
    $this->sandbox = $sandbox;
  }

  public function paymentRequest(string $description, string $callbackURL, string $email = '', string $mobile = ''): string {
    $apiUrl = $this->getSandbox() ? static::API_REQUEST_SANDBOX : static::API_REQUEST_LIVE;
    
    // Ensure metadata fields are always strings
    $metadata = [];
    if (!empty($mobile)) {
      $metadata['mobile'] = (string) $mobile;
    }
    if (!empty($email)) {
      $metadata['email'] = (string) $email;
    }

    $data = [
      'merchant_id' => $this->merchantID,
      'amount' => $this->amount,
      'callback_url' => $callbackURL,
      'description' => $description,
    ];

    if (!empty($metadata)) {
      $data['metadata'] = $metadata;
    }

    $response = $this->sendRequest($apiUrl, $data);
    
    if (isset($response['data']['code']) && $response['data']['code'] == 100) {
      return $response['data']['authority'];
    }
    
    throw new ZarinpalException($this->t('Payment request failed: @message', ['@message' => $response['errors']['message'] ?? 'Unknown error']));
  }

  public function verifyTransaction(string $authority): string {
    $apiUrl = $this->getSandbox() ? static::API_VERIFY_SANDBOX : static::API_VERIFY_LIVE;
    
    $data = [
      'merchant_id' => $this->merchantID,
      'amount' => $this->amount,
      'authority' => $authority,
    ];

    $response = $this->sendRequest($apiUrl, $data);
    
    if (isset($response['data']['code']) && $response['data']['code'] == 100) {
      return $response['data']['ref_id'];
    }
    
    throw new ZarinpalExceptionVerification($this->t('Transaction verification failed: @message', ['@message' => $response['errors']['message'] ?? 'Unknown error']));
  }

  public function buildGateURL(string $authority): string {
    return ($this->getSandbox() ? static::GATEWAY_URL_SANDBOX : static::GATEWAY_URL) . $authority;
  }

  private function sendRequest(string $url, array $data): array {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_USERAGENT, 'ZarinPal Rest API v4');
    curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'POST');
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    
    $result = curl_exec($ch);
    curl_close($ch);
    
    return json_decode($result, true) ?? [];
  }
}